#include "xml2svg.hpp"
#include <fstream>
using SixtePoint = sixte::SixtePoint;

// Constructor to initialize xml2svg parameters
xml2svg_parameters::xml2svg_parameters() {
    XMLFile = sixte::queryParameterString("XMLFile");
    SVGName = sixte::queryParameterString("SVGName");
    std::string writeIDInput = sixte::queryParameterString("WriteID");
    
    try {
        SVGWidth = sixte::queryParameterDouble("SVGWidth");
    } catch (...) {
        SVGWidth = 200.0;
    }

    try {
        Border = sixte::queryParameterDouble("Border");
    } catch (...) {
        Border = 10.0;
    }

    if (writeIDInput.empty()) {
        writeIDInput = "no";
    }
    writeID = (writeIDInput == "yes");
    
    try {
        DrawN = sixte::queryParameterInt("DrawN");
    } catch (...) {
        DrawN = -1;  // Default value -1 to draw all pixels
    }
}

void exportToSVG(const std::vector<std::vector<sixte::Rectangle2d>>& chip_rectangles,
                 const std::vector<std::unique_ptr<sixte::ArrayGeometry>>& arrgeometries,
                 const std::vector<std::unique_ptr<sixte::Geometry>>& geometries,
                 const std::string& filename,
                 bool writeID, double svgWidth, double border, int DrawN) {
    std::ofstream svgFile(filename);
    
    // SVG header
    svgFile << "<svg xmlns=\"http://www.w3.org/2000/svg\" width=\"" << svgWidth
            << "\" height=\"" << svgWidth << "\">\n";

    size_t total_chips = chip_rectangles.size();

    // Calculate global limits
    double global_minX = std::numeric_limits<double>::max();
    double global_maxX = std::numeric_limits<double>::lowest();
    double global_minY = std::numeric_limits<double>::max();
    double global_maxY = std::numeric_limits<double>::lowest();

     for (size_t pos = 0; pos < chip_rectangles.size(); ++pos) {
        const auto& array_geometry = arrgeometries[pos];
        const auto& chip_geometry = geometries[pos];
        
        sixte::BoundingBox bbox = array_geometry->boundingBox();
        
        double localMinX = bbox.bottom_left().x();
        double localMinY = bbox.bottom_left().y();
        double localMaxX = bbox.top_right().x();
        double localMaxY = bbox.top_right().y();

        SixtePoint bottomLeftDet(localMinX, localMinY, 0);
        SixtePoint bottomRightDet(localMaxX, localMinY, 0);
        SixtePoint topLeftDet(localMinX, localMaxY, 0);
        SixtePoint topRightDet(localMaxX, localMaxY, 0);
    
        SixtePoint bottomLeftFocal = chip_geometry->transformDetToFocal(bottomLeftDet);
        SixtePoint bottomRightFocal = chip_geometry->transformDetToFocal(bottomRightDet);
        SixtePoint topLeftFocal = chip_geometry->transformDetToFocal(topLeftDet);
        SixtePoint topRightFocal = chip_geometry->transformDetToFocal(topRightDet);

        double focal_minX = std::min({bottomLeftFocal.x(), bottomRightFocal.x(),
                                      topLeftFocal.x(), topRightFocal.x()});
        double focal_minY = std::min({bottomLeftFocal.y(), bottomRightFocal.y(),
                                      topLeftFocal.y(), topRightFocal.y()});
        double focal_maxX = std::max({bottomLeftFocal.x(), bottomRightFocal.x(),
                                      topLeftFocal.x(), topRightFocal.x()});
        double focal_maxY = std::max({bottomLeftFocal.y(), bottomRightFocal.y(),
                                      topLeftFocal.y(), topRightFocal.y()});
    
        global_minX = std::min(global_minX, focal_minX);
        global_minY = std::min(global_minY, focal_minY);
        global_maxX = std::max(global_maxX, focal_maxX);
        global_maxY = std::max(global_maxY, focal_maxY);
    }

    if (DrawN != -1) {
        double worldBorder = border * (global_maxX - global_minX) / svgWidth;
        double worldMinX = global_minX - worldBorder;
        double worldMaxX = global_maxX + worldBorder;
        double worldMinY = global_minY - worldBorder;
        double worldMaxY = global_maxY + worldBorder;

        double scaleX = svgWidth / (worldMaxX - worldMinX);
        double scaleY = svgWidth / (worldMaxY - worldMinY);
        double scale = std::min(scaleX, scaleY);
        
        double globalBottomLeftX = std::numeric_limits<double>::max();
        double globalBottomLeftY = std::numeric_limits<double>::max();
        double globalTopRightX = std::numeric_limits<double>::lowest();
        double globalTopRightY = std::numeric_limits<double>::lowest();

        // Draw each chip and its pixels
        for (size_t pos = 0; pos < chip_rectangles.size(); ++pos) {
            const auto& chip = chip_rectangles[pos];
            const auto& array_geometry = arrgeometries[pos];
            const auto& chip_geometry = geometries[pos];
        
            sixte::BoundingBox bbox = array_geometry->boundingBox();

            double localMinX = bbox.bottom_left().x();
            double localMaxX = bbox.top_right().x();
            double localMinY = bbox.bottom_left().y();
            double localMaxY = bbox.top_right().y();

            SixtePoint bottomLeftDet(localMinX, localMinY, 0);
            SixtePoint bottomRightDet(localMaxX, localMinY, 0);
            SixtePoint topLeftDet(localMinX, localMaxY, 0);
            SixtePoint topRightDet(localMaxX, localMaxY, 0);
        
            SixtePoint bottomLeftFocal = chip_geometry->transformDetToFocal(bottomLeftDet);
            SixtePoint bottomRightFocal = chip_geometry->transformDetToFocal(bottomRightDet);
            SixtePoint topLeftFocal = chip_geometry->transformDetToFocal(topLeftDet);
            SixtePoint topRightFocal = chip_geometry->transformDetToFocal(topRightDet);
        
            // bounding box in focal plane coordinates
            double focalMinX = std::min({
                bottomLeftFocal.x(),
                bottomRightFocal.x(),
                topLeftFocal.x(),
                topRightFocal.x()
            });
        
            double focalMaxX = std::max({
                bottomLeftFocal.x(),
                bottomRightFocal.x(),
                topLeftFocal.x(),
                topRightFocal.x()
            });
        
            double focalMinY = std::min({
                bottomLeftFocal.y(),
                bottomRightFocal.y(),
                topLeftFocal.y(),
                topRightFocal.y()
            });
        
            double focalMaxY = std::max({
                bottomLeftFocal.y(),
                bottomRightFocal.y(),
                topLeftFocal.y(),
                topRightFocal.y()
            });

            globalBottomLeftX = std::min(globalBottomLeftX, focalMinX);
            globalBottomLeftY = std::min(globalBottomLeftY, focalMinY);
            globalTopRightX = std::max(globalTopRightX, focalMaxX);
            globalTopRightY = std::max(globalTopRightY, focalMaxY);
        
            svgFile << "<polygon points=\""
                    << (bottomLeftFocal.x()-worldMinX)*scale      << "," << svgWidth-((bottomLeftFocal.y()-worldMinY)*scale) << " "
                    << (bottomRightFocal.x()-worldMinX)*scale     << "," << svgWidth-((bottomRightFocal.y()-worldMinY)*scale) << " "
                    << (topRightFocal.x()-worldMinX)*scale        << "," << svgWidth-((topRightFocal.y()-worldMinY)*scale)  << " "
                    << (topLeftFocal.x()-worldMinX)*scale         << "," << svgWidth-((topLeftFocal.y()-worldMinY)*scale)
                    << "\" style=\"fill:#ffeeaa;stroke:black;stroke-width:1\"/>";
 

            size_t pixelsToDraw = (DrawN == -1) ? chip.size() : std::min<size_t>(chip.size(), static_cast<size_t>(DrawN));

            // Draw pixels
            for (size_t j = 0; j < pixelsToDraw; ++j) {
                const auto& rect = chip[j];
                
                double pixel_minX = rect.bottom_left().x();
                double pixel_maxX = rect.top_right().x();
                double pixel_minY = rect.bottom_left().y();
                double pixel_maxY = rect.top_right().y();
                
                SixtePoint bottomLeftDet_pix(pixel_minX, pixel_minY, 0);
                SixtePoint bottomRightDet_pix(pixel_maxX, pixel_minY, 0);
                SixtePoint topLeftDet_pix(pixel_minX, pixel_maxY, 0);
                SixtePoint topRightDet_pix(pixel_maxX, pixel_maxY, 0);

                SixtePoint bottomLeftFocal_pix = chip_geometry->transformDetToFocal(bottomLeftDet_pix);
                SixtePoint bottomRightFocal_pix = chip_geometry->transformDetToFocal(bottomRightDet_pix);
                SixtePoint topLeftFocal_pix = chip_geometry->transformDetToFocal(topLeftDet_pix);
                SixtePoint topRightFocal_pix = chip_geometry->transformDetToFocal(topRightDet_pix);
            
                double x1 = (bottomLeftFocal_pix.x() - worldMinX) * scale;
                double y1 = svgWidth - ((bottomLeftFocal_pix.y() - worldMinY) * scale);
                double x2 = (bottomRightFocal_pix.x() - worldMinX) * scale;
                double y2 = svgWidth - ((bottomRightFocal_pix.y() - worldMinY) * scale);
                double x3 = (topRightFocal_pix.x() - worldMinX) * scale;
                double y3 = svgWidth - ((topRightFocal_pix.y() - worldMinY) * scale);
                double x4 = (topLeftFocal_pix.x() - worldMinX) * scale;
                double y4 = svgWidth - ((topLeftFocal_pix.y() - worldMinY) * scale);
            
                svgFile << "  <polygon points=\""
                        << x1 << "," << y1 << " "
                        << x2 << "," << y2 << " "
                        << x3 << "," << y3 << " "
                        << x4 << "," << y4 << "\" "
                        << "style=\"fill:none;stroke:red;stroke-width:0.5\"/>";
            if (writeID) {
                double text_x = (x1 + x3) / 2;
                double text_y = (y1 + y3) / 2;
                svgFile << "  <text x=\"" << text_x << "\" y=\"" << text_y
                        << "\" font-family=\"Verdana\" font-size=\"" << (x2 - x1) / 3
                        << "\" fill=\"black\" text-anchor=\"middle\" dominant-baseline=\"middle\">"
                        << (j) << "</text>\n";
            }
        
            }
            
            // Draw focal plane axes
            double L = std::max(global_maxX - global_minX, global_maxY - global_minY);

            SixtePoint x1_det(-L, 0, 0);
            SixtePoint x2_det( L, 0, 0);
            SixtePoint y1_det(0, -L, 0);
            SixtePoint y2_det(0,  L, 0);

            SixtePoint x1_focal = chip_geometry->transformDetToFocal(x1_det);
            SixtePoint x2_focal = chip_geometry->transformDetToFocal(x2_det);
            SixtePoint y1_focal = chip_geometry->transformDetToFocal(y1_det);
            SixtePoint y2_focal = chip_geometry->transformDetToFocal(y2_det);

            double x1_svg = (x1_focal.x() - worldMinX) * scale;
            double y1_svg = svgWidth - ((x1_focal.y() - worldMinY) * scale);
            double x2_svg = (x2_focal.x() - worldMinX) * scale;
            double y2_svg = svgWidth - ((x2_focal.y() - worldMinY) * scale);
            double y1_x_svg = (y1_focal.x() - worldMinX) * scale;
            double y1_y_svg = svgWidth - ((y1_focal.y() - worldMinY) * scale);
            double y2_x_svg = (y2_focal.x() - worldMinX) * scale;
            double y2_y_svg = svgWidth - ((y2_focal.y() - worldMinY) * scale);

            svgFile << "<line x1=\"" << x1_svg << "\" y1=\"" << y1_svg
                    << "\" x2=\"" << x2_svg << "\" y2=\"" << y2_svg
                    << "\" style=\"stroke:grey;stroke-width:1\" />";
            svgFile << "<line x1=\"" << y1_x_svg << "\" y1=\"" << y1_y_svg
                    << "\" x2=\"" << y2_x_svg << "\" y2=\"" << y2_y_svg
                    << "\" style=\"stroke:grey;stroke-width:1\" />";

        }

    }else {
           double worldBorder = border * (global_maxX - global_minX) / svgWidth;
           double worldMinX = global_minX - worldBorder;
           double worldMaxX = global_maxX + worldBorder;
           double worldMinY = global_minY - worldBorder;
           double worldMaxY = global_maxY + worldBorder;

           double scaleX = svgWidth / (worldMaxX - worldMinX);
           double scaleY = svgWidth / (worldMaxY - worldMinY);
           double scale = std::min(scaleX, scaleY);


           for (size_t chip_idx = 0; chip_idx < total_chips; ++chip_idx) {
               const auto& chip = chip_rectangles[chip_idx];
               for (size_t j = 0; j < chip.size(); ++j) {
                   const auto& rect = chip[j];
                   auto bottom_left = rect.bottom_left();
                   auto top_right = rect.top_right();
                   
                   double x = (bottom_left.x() - worldMinX) * scale;
                   double y = svgWidth - (top_right.y() - worldMinY) * scale;
                   double width = (top_right.x() - bottom_left.x()) * scale;
                   double height = (top_right.y() - bottom_left.y()) * scale;
                   
                   svgFile << "  <rect x=\"" << x << "\" y=\"" << y
                           << "\" width=\"" << width << "\" height=\"" << height
                           << "\" style=\"fill:none;stroke:red;stroke-width:0.5\"/>\n";
                   
                   if (writeID) {
                       double text_x = x + width / 2;
                       double text_y = y + height / 2;
                       svgFile << "  <text x=\"" << text_x << "\" y=\"" << text_y
                               << "\" font-family=\"Verdana\" font-size=\"" << width / 3
                               << "\" fill=\"black\" text-anchor=\"middle\" dominant-baseline=\"middle\">"
                               << (j) << "</text>\n";
                   }
               }
           }
        double xAxisY_svg, yAxisX_svg;
        xAxisY_svg = svgWidth - ((0.0 - worldMinY) * scale);
        yAxisX_svg = (0.0 - worldMinX) * scale;
        svgFile << "<line x1=\"" << 0 << "\" y1=\"" << xAxisY_svg
                << "\" x2=\"" << svgWidth << "\" y2=\"" << xAxisY_svg
                << "\" style=\"stroke:grey;stroke-width:1\" />";
        svgFile << "<line x1=\"" << yAxisX_svg << "\" y1=\"" << 0
                << "\" x2=\"" << yAxisX_svg << "\" y2=\"" << svgWidth
                << "\" style=\"stroke:grey;stroke-width:1\" />";
       }
       svgFile << "</svg>\n";
       svgFile.close();
       std::cout << "SVG file generated: " << filename << std::endl;
   }

int xml2svg_main() {
    xml2svg_parameters par;
    int status = EXIT_SUCCESS;

    set_toolname("xml2svg");
    set_toolversion("0.1");

    do {
        // Load the XML files
        auto xmlfiles = sixte::loadXMLFiles();
        std::string xml_path = sixte::loadXMLPath();
        std::vector<sixte::XMLData> xmls;
        xmls.reserve(xmlfiles.xml_documents.size());

        // Parse the XML documents
        for (auto& f : xmlfiles.xml_documents) {
            xmls.emplace_back(f, xml_path);
        }

        std::vector<std::unique_ptr<sixte::ArrayGeometry>> arrgeometries;
        std::vector<std::unique_ptr<sixte::Geometry>> geometries;  // Store the geometries with WCS
        arrgeometries.reserve(xmls.size());
        geometries.reserve(xmls.size());

        for (auto& xml : xmls) {
            arrgeometries.push_back(sixte::createGeometry(xml));
            geometries.push_back(std::make_unique<sixte::Geometry>(xml, arrgeometries.back()->boundingBox()));
        }
        
        // Store the rectangles by chip
        std::vector<std::vector<sixte::Rectangle2d>> chip_rectangles;
        chip_rectangles.reserve(arrgeometries.size());
        

        for (size_t i = 0; i < arrgeometries.size(); ++i) {
            size_t num_pixels = arrgeometries[i]->numpix();
            headas_chat(3, "Geometry %zu contains %zu pixels.\n", i + 1, num_pixels);
            std::vector<sixte::Rectangle2d> current_chip;
            current_chip.reserve(num_pixels);
            for (size_t pix_id = 0; pix_id < num_pixels; ++pix_id) {
                current_chip.push_back(arrgeometries[i]->getPolygon(pix_id));
            }
            chip_rectangles.emplace_back(std::move(current_chip));
        }

        exportToSVG(chip_rectangles, arrgeometries, geometries, par.SVGName, par.writeID, par.SVGWidth, par.Border, par.DrawN);


    } while (false);

    headas_chat(3, "cleaning up ...\n");
    if (status == EXIT_SUCCESS) headas_chat(3, "completed successfully!\n\n");

    return status;
}
