/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.

   Copyright 2020 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "sixtesim.h"

#define TOOLSUB sixtesim_main

#include "NewSixt_main.cpp"
#include "SixteException.h"
#include "TemplateFitsFiles.h"

int sixtesim_main() {
  // Register HEAdas task
  set_toolname("sixtesim");
  set_toolversion("0.5");

  try {
    // ---------- Initialization ----------
    healog(3) << "initialize ...\n";
    
    sixte::SimulationParameters sim_params;
    sixte::initSixteRng(sim_params.seed);

    std::vector<sixte::XMLData> xml_datas;
    std::vector<sixte::XMLData> xml_tel_datas;
    std::vector<sixte::ObsInfo> obs_infos;
    std::vector<sixte::ObsInfo> obs_tel_infos;
    sixte::GTICollection gti_collection(sim_params.input_files.gti_file,
                                        sim_params.time);

    size_t num_xmls = sim_params.input_files.num_xmls;
    for (size_t current_xml=0; current_xml<num_xmls; current_xml++){
      xml_datas.emplace_back(sim_params.input_files.xml_files[current_xml], sim_params.input_files.xml_path);
      obs_infos.emplace_back(xml_datas[current_xml], sim_params.pointing, gti_collection);
    }
    
    size_t num_tel_xmls = sim_params.input_files.xml_tel_files.size();
    for (size_t current_xml=0; current_xml<num_tel_xmls; current_xml++){
      xml_tel_datas.emplace_back(sim_params.input_files.xml_tel_files[current_xml], sim_params.input_files.xml_path);
      obs_tel_infos.emplace_back(xml_tel_datas[current_xml], sim_params.pointing, gti_collection);
    }
    
    sixte::Simulator Simulator(sim_params, xml_datas, xml_tel_datas, obs_infos, obs_tel_infos, gti_collection);

    // ---------- Simulation  ----------
    healog(3) << "\nstart simulation ...\n";
    Simulator.runSimulation();
//    for(auto obs_info : obs_infos) {
//     obs_info.~ObsInfo();
//    }
  } catch (const std::exception& e) {
    sixte::printError(e.what());
    sixt_destroy_rng();
    return EXIT_FAILURE;
  } catch (const CCfits::FitsException& e) {
    sixte::printError(e.message());
    return EXIT_FAILURE;
  }
  // ---------- Clean up ----------
  healog(3) << "\ncleaning up ...\n";
  sixt_destroy_rng();
  healog(3) << "finished successfully!\n\n";

  return EXIT_SUCCESS;
}

sixte::Simulator::Simulator(sixte::SimulationParameters& sim_params,
                            std::vector<sixte::XMLData>& xml_datas,
                            std::vector<sixte::XMLData>& xml_tel_datas,
                            std::vector<ObsInfo>& obs_infos,
                            const std::vector<sixte::ObsInfo>& obs_tel_infos,
                            const GTICollection& gti_collection)
  : attitude_(sim_params.pointing, sim_params.time, sim_params.attitude_dt),
    gti_collection_(gti_collection),
    obs_infos_(obs_infos),
//    num_xmls_(sim_params.input_files.num_xmls),
    num_tels_(sim_params.input_files.num_tels),
    num_chips_(sim_params.input_files.num_chips),
    photon_generation_(sim_params.output_files.photon_lists, sim_params.output_files.clobber,
                       sim_params.input_files.simput_files, xml_tel_datas, obs_tel_infos),
    show_progress_(sim_params.show_progress) {
      // TODO: - Clean up constructor (resize, nested loops, ...)
      if (!xml_datas.empty() &&
          xml_datas[0].root().attributeAsString("telescop") == "THESEUS" &&
          xml_datas[0].root().attributeAsString("instrume") == "SXI") {
        isTheseus_ = true;
      }
      auto rmf_registry = std::make_shared<sixte::RmfRegistry>();
      detector_.resize(num_tels_);
      if (sim_params.background) {
        pha_background_.resize(num_tels_);
        aux_background_.resize(num_tels_);
      }

      carrier_buffer_max_size_ = 1;
      size_t xml_count = 0;
      for (size_t tel_id=0; tel_id<num_tels_; tel_id++) {
        photon_imaging_.emplace_back(sim_params.output_files.impact_lists[tel_id],
                                     sim_params.output_files.clobber, xml_tel_datas[tel_id], obs_tel_infos[tel_id]);

        for (size_t chip_id=0; chip_id<num_chips_[tel_id]; chip_id++) {
          detector_[tel_id].emplace_back(xml_datas[xml_count], sim_params.output_files, obs_infos[xml_count],
                                         sim_params.skip_invalids, xml_count, chip_id, rmf_registry);

          if (sim_params.background) {
            auto detector = xml_datas[tel_id].child("detector");
            bool has_phabackground = detector.hasChild("phabackground");
            bool has_auxbackground = detector.hasChild("auxbackground");
            
            if (has_phabackground && has_auxbackground) {
              throw SixteException("Both PHA and AUX Background file given in xml, just one allowed");
            }

            carrier_buffer_max_size_++;

            if (has_phabackground) {
              pha_background_[tel_id].emplace_back(xml_datas[xml_count],
                                                   detector_[tel_id][chip_id].sensorGeometry(),
                                                   detector_[tel_id][chip_id].canonicalEbounds(),
                                                   tel_id, chip_id);

            } else if (has_auxbackground) {
              auto auxbkg = detector.child("auxbackground");
              auto aux_background = auxbkg.attributeAsString("filename");
              auto aux_rate = auxbkg.attributeAsDouble("rate");
              auto chip_area = detector_[tel_id][chip_id].sensorGeometry()->totalSurfaceArea();

              std::optional<std::string> simput_lc = std::nullopt;
              if (auxbkg.hasAttribute("lightcurve")) {
                simput_lc = auxbkg.attributeAsString("lightcurve");
              }
              aux_background_[tel_id].emplace_back(aux_background, chip_area, tel_id, chip_id, aux_rate, simput_lc);
              aux_background_initialized_ = true;

            } else {
              printWarning("No Background file found, continuing without Background");
            }
          }
          xml_count ++;
        }
        //TODO: telescope_module_[tel_id];

      }
}


void sixte::Simulator::runSimulation() {
  runObservation();
  runPostprocessing();
}

void sixte::Simulator::setDetectorStartTime(double tstart) {
  for (auto& det: detector_) {
    for (auto& chip: det) {
      chip.setSensorStartTime(tstart);
    }
  }
}

void sixte::Simulator::runObservation() {
  Progressbar progressbar(show_progress_, obs_infos_[0].total_sim_time);
  double simulated_time = 0.;

  for (auto gti_bin: gti_collection_.getAllGtiBins()) {
    setDetectorStartTime(gti_bin.first);

    sixte::Carriers carrier_buffer;

    if (auto src_photon_carrier = getNextSrcPhotonCarrier(gti_bin)) {
      carrier_buffer.push(src_photon_carrier);
    }

    for (auto& pha_background_tm: pha_background_) {
      for (auto& pha_background_chip: pha_background_tm) {
        if (auto bkg_carrier = pha_background_chip.getNextBkgCarrier(gti_bin)) {
          carrier_buffer.push(bkg_carrier);
        }
      }
    }

    if (aux_background_initialized_) {
      for (size_t tel = 0; tel < num_tels_; tel++) {
        for (size_t chip = 0; chip < num_chips_[tel]; chip++) {
          if (auto bkg_carrier = aux_background_[tel][chip]
              .getNextAUXBkgCarrier(detector_[tel][chip], gti_bin, attitude_.dt())) {
            carrier_buffer.push(bkg_carrier);
          }
        }
      }
    }

    for (;;) {
      if (carrier_buffer.size() > carrier_buffer_max_size_)
        throw SixteException("More than " + std::to_string(carrier_buffer_max_size_) + " events in carrier buffer");

      if (carrier_buffer.empty()) break;

      auto carrier = sixte::pop_next_carrier(carrier_buffer);

      size_t carrier_telid = carrier->photon_metainfo().tel_id_;
      size_t carrier_chipid = carrier->photon_metainfo().chip_id_;

      detector_[carrier_telid][carrier_chipid].propagateReadout(carrier->creationTime());
      detector_[carrier_telid][carrier_chipid].addCarrier(*carrier);

      progressbar.update(carrier->creationTime() - gti_bin.first + simulated_time);

      auto next_carrier = getNextCarrierOfType(carrier->photon_metainfo(), gti_bin);

      if (next_carrier) carrier_buffer.push(next_carrier);
    }

    assert(carrier_buffer.empty());

    for (auto& det: detector_) {
      for (auto& chip: det) {
        chip.finishSimulation(gti_bin.second);
        chip.clearSensor();
      }
    }
    simulated_time += gti_bin.second-gti_bin.first;
  }

  for (auto& photon_imaging : photon_imaging_) {
    photon_imaging.checkIfImaged();
  }

  progressbar.finish();
}

std::shared_ptr<sixte::Carrier> sixte::Simulator::getNextCarrierOfType(const PhotonMetainfo& photon_metainfo,
                                                                       std::pair<double, double> dt) {
  auto src_type = (SrcType) photon_metainfo.src_id_;
  if (src_type > 0) {
    return sixte::Simulator::getNextSrcPhotonCarrier(dt);
  } else if (src_type == SrcType::pha_bkg) {
    return pha_background_[photon_metainfo.tel_id_][photon_metainfo.chip_id_].getNextBkgCarrier(dt);
  } else if (src_type == SrcType::aux_bkg) {
    return aux_background_[photon_metainfo.tel_id_][photon_metainfo.chip_id_]
      .getNextAUXBkgCarrier(detector_[photon_metainfo.tel_id_][photon_metainfo.chip_id_], dt, attitude_.dt());
  } else if (src_type == SrcType::mxs) {
    throw SixteException("MXS type not implemented yet");
  } else {
    throw SixteException("Unknown source type");
  }
}

sixte::CarrierPtr sixte::Simulator::getNextSrcPhotonCarrier(std::pair<double, double> dt) {
  static Carriers carrier_buffer{};
  
  // Return the next carrier. Otherwise, loop over photon generation, imaging,
  // and absorption, until new carriers are created.
  for (;;) {
    
    if (!carrier_buffer.empty()) return pop_next_carrier(carrier_buffer);

    auto src_photon = photon_generation_.getNextPhoton(attitude_, dt);
    
    if (!src_photon) return nullptr;
    
    size_t photon_telid = src_photon->photon_metainfo().tel_id_;
    auto src_photon_impact = photon_imaging_[photon_telid].doImaging(attitude_, *src_photon);

    if (!src_photon_impact) continue;

    if (isTheseus_) {
      auto src_photon_carriers = detector_[photon_telid][src_photon_impact->photon_metainfo().chip_id_].absorb(*src_photon_impact);
      transferCarriers(src_photon_carriers, carrier_buffer);
    } else {
    for (auto& det : detector_[photon_telid]) {
      auto src_photon_carriers = det.absorb(*src_photon_impact);
      transferCarriers(src_photon_carriers, carrier_buffer);
    }
  }
}
}

void sixte::Simulator::runPostprocessing() {
  for (auto& det : detector_) {
    for (auto& chip : det) {
      chip.postprocessing(attitude_, gti_collection_);
    }
  }
}

// ToDo: create extra tool ero_calevents
