/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.

   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <string>
#include <vector>
#include <Parameters.h>
#include "PhotonImaging.h"
#include "sixte_random.h"
#include "SixteProgressbar.h"
#include "XMLData.h"
#include "ArrayGeometry.h"
#include "NewAttitude.h"
#include "NewSourceCatalog.h"
#include "SixteLinterp.h"
#include "NewArf.h"

// TODO use new ARF class
#include "arf.h"
#include "simput.h"
extern "C" {
  #include "wcs.h"
  #include "region.h"
}

namespace sixte {

class Parameters {
 public:
  Parameters();

  std::string ARFCorr;    // Corrected ARF output file
  std::string XMLFile;    // XML input file with instrument definition
  int Seed;              // Seed for RNG
  bool clobber;           // Overwrite output files if exist?

  std::string attitude_file;
  std::string gti_file;
  ObsPointing obspointing;
  ObsTime obstime;

  double SourceRA;       // Right ascension of source [degree]
  double SourceDec;      // Declination of source [degree]
  std::string simputfile; // name of the source catalog

  double RefRA;           // Right ascension of WCS reference point [deg]
  double RefDec;          // Declination of WCS reference point [deg]
  std::string Projection; // WCS projection type (usually SIN)
  // Further WCS keys - UNIT is assumed to be deg
  double crpix1;
  double crpix2;
  double cdelt1;
  double cdelt2;

  std::string imgfile;   // If given, WCS is read from here instead of command line

  std::string regfilter;  // Region filter filename

  unsigned int n_photons;         // Number of photons simulated per ARF bin
  unsigned int arf_sampling_factor;         // Sample every Nth bin

  bool show_progress;
};

struct GeoPair {
  std::unique_ptr<Geometry> abs;
  std::unique_ptr<ArrayGeometry> arr;
};

class ARFCorr {
 public:
  ARFCorr(const struct ARF *const arf, std::vector<sixte::XMLData>& xml_datas, const Parameters &pars);

  void applyCorrection(const std::string& arfpath, const struct ARF *const arf,
                       const Parameters &pars);

  // calculate ARF correction using a single point source
  void calcCorrection(
    const std::pair<double,double>& srcpos,
    const NewArf& arf,
    unsigned int num_photons,
    SAORegion* region, wcsprm& wcspr, const Parameters &pars);

  // calculate ARF correction using a SIMPUT catalog
  void calcCorrection(
    const std::string& catalog_file,
    const NewArf& arf,
    unsigned int num_photons,
    SAORegion* region, wcsprm& wcspr, const Parameters &pars);

  bool testPhoton(SixtePhoton& phot, SAORegion* region, wcsprm& wcspr);

  std::vector<unsigned int> bin_idxs;       // ARF bins for which correction factors are calculated
  std::vector<double> corr_fac;    // Associated correction factors
  std::vector<double> corr_energ;  // for specified energies (bin midpoints).

  GTICollection gtis;
  NewAttitude att;
  double focal_length;
  std::vector<GeoPair> geometries;
  ObsInfo obsinfo;
  PhotonImaging tel;
  bool is_theseus_{false};
};

}
