/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                       Erlangen-Nuernberg
*/

#include "healog.h"
#include "longnam.h"
#include "paraminput.h"
#include "sixte_random.h"
#include "SixteException.h"
#include "XMLData.h"
#include "ArrayGeometry.h"
#include "NewAttitude.h"
#include <cctype>
#include "SixteCCFits.h"


#define TOOLSUB projev_main
#include "sixt_main.c"

class projev_parameters {
 public:

  projev_parameters() {
    EvtFile = sixte::queryParameterString("RawData");
    XMLFile = sixte::queryParameterString("XMLFile");
    AttitudeFile = sixte::queryParameterString("Attitude");

    RA = sixte::queryParameterDouble("RA");
    Dec = sixte::queryParameterDouble("Dec");
    rollangle = sixte::queryParameterDouble("Rollangle");
    MJDREF = sixte::queryParameterDouble("MJDREF");
    TSTART = sixte::queryParameterDouble("TSTART");
    Exposure = sixte::queryParameterDouble("Exposure");

    chatter = sixte::queryParameterInt("chatter");
    Seed = sixte::queryParameterInt("Seed");
  }
  std::string EvtFile;
  std::string XMLFile;
  std::string AttitudeFile;

  double RA, Dec, rollangle;

  double MJDREF;
  double TSTART;
  double Exposure;

  int Seed;
  int chatter;
};


int projev_main() {
  // Register HEAdas task
  set_toolname("projev");
  set_toolversion("0.1");
  int status = EXIT_SUCCESS;

  try {
    // ---------- Initialization ----------
    healog(3) << "initialize ...\n";
    projev_parameters sim_params;
    sixte::initSixteRng(sim_params.Seed);

    sixte::XMLData xml_data(sim_params.XMLFile);
    std::unique_ptr<sixte::ArrayGeometry> arr_geometry = sixte::createGeometry(xml_data);
    sixte::Geometry abs_geometry(xml_data);

    // set up attitude
    sixte::ObsPointing op;
    sixte::ObsTime ot;
    sixte::NewAttitude ac(op,ot, 1.);

    // open FITS file
    fitsfile *fptr;
    fits_open_file(&fptr, sim_params.EvtFile.c_str(), READWRITE, &status);
    sixte::checkStatusThrow(status, "Failed opening event file");

    double focal_length = xml_data.child("telescope").child("focallength").attributeAsDouble("value");

    // ---------- Main Work ----------
    healog(3) << "\nrun photon projection ...\n";
    arr_geometry->doPhotonProjection(ac, abs_geometry, focal_length, fptr,
        sim_params.TSTART, sim_params.TSTART+sim_params.Exposure);

    // ---------- Clean up ----------
    healog(3) << "\ncleaning up ...\n";
    fits_close_file(fptr, &status);
    sixte::checkStatusThrow(status, "Failed closing event file");

  } catch (const std::exception& e) {
    sixte::printError(e.what());
    sixt_destroy_rng();
    return EXIT_FAILURE;
  }

  sixt_destroy_rng();

  healog(3) << "finished successfully!\n\n";
  return EXIT_SUCCESS;
}

