/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.

   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "piximpacts.hpp"
#include "ArrayGeometry.h"
#include "XMLData.h"

#define TOOLSUB piximpacts_main
#include "sixt_main.c"

int piximpacts_main() {
  // Register HEAdas task
  set_toolname("piximpacts");
  set_toolversion("1.0");

  try {
    // ---------- Initialization ----------
    healog(3) << "initialize ...\n";
    sixte::Parameters sim_params;

    sixte::XMLData xml_data(sim_params.xml_file);

    // ---------- Simulation  ----------
    
    // set up input file
    sixte::Fitsfile impactlist(sim_params.impact_file, sixte::FileMode::read);
    impactlist.moveToExt("IMPACTS");

    // get geometry
    sixte::Geometry fp_geometry(xml_data);
    std::unique_ptr<sixte::ArrayGeometry> arr_geometry = sixte::createGeometry(xml_data);

    // set up output file
    std::unique_ptr<CCfits::FITS> ccf_tmp = sixte::sixteOpenFITSFileWrite(
        sim_params.piximpact_file,
        sim_params.clobber,
        true,
        "piximpact file");
    ccf_tmp->destroy();

    sixte::Fitsfile piximpactlist(sim_params.piximpact_file, sixte::FileMode::read);
    piximpactlist.moveToExt("PIXELIMPACT");

    // assign all the impacts
    size_t num_imp = impactlist.getNumRows();

    size_t write_row = 1;

    for (size_t ii=0; ii<num_imp; ii++) {
      size_t read_row = ii+1;
      auto imp = sixte::getImpact(impactlist, read_row).value();

      // project to focal plane
      auto fp_pos = imp.detector_position();

      if (!fp_pos.has_value()) continue; 

      // get_position on detector and pixid
      auto arr_pos = fp_geometry.transformFocalToDet(fp_pos.value());
      auto pixid = arr_geometry->getPixId({arr_pos.x(), arr_pos.y()});

      if (!pixid.has_value()) continue;

      // write
      piximpactlist.writeCol("TIME", write_row, imp.time());
      piximpactlist.writeCol("ENERGY", write_row, imp.energy());
      piximpactlist.writeCol("X", write_row, arr_pos.x());
      piximpactlist.writeCol("Y", write_row, arr_pos.y());
      piximpactlist.writeCol("PH_ID", write_row, imp.photon_metainfo().ph_id_);
      piximpactlist.writeCol("SRC_ID", write_row, imp.photon_metainfo().src_id_);
      piximpactlist.writeCol("PIXID", write_row, pixid.value()+1); // one-based!
      piximpactlist.writeCol("U", write_row, 0);
      piximpactlist.writeCol("V", write_row, 0);

      write_row++;
    }

  } catch (const std::exception& e) {
    sixte::printError(e.what());
    return EXIT_FAILURE;
  }

  // ---------- Clean up ----------
  healog(3) << "\ncleaning up ...\n";


  healog(3) << "finished successfully!\n\n";
  return EXIT_SUCCESS;
}

namespace sixte {
std::optional<SixtePhoton> getImpact(sixte::Fitsfile& infile, size_t row) {

  if (row > infile.getNumRows()) {
    return std::nullopt;
  }

  double time, x, y;
  float energy;
  int ph_id, src_id;

  infile.readCol("TIME", row, time);
  infile.readCol("ENERGY", row, energy);
  infile.readCol("X", row, x);
  infile.readCol("Y", row, y);
  infile.readCol("PH_ID", row, ph_id);
  infile.readCol("SRC_ID", row, src_id);

  PhotonMetainfo meta(ph_id, src_id);
  SixtePoint pt(x, y, 0);
  SixtePhoton phot(time, energy, pt, meta);

  return phot;
}
}
