/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.

   Copyright 2020 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "phoimg.hpp"

#define TOOLSUB phoimg_main
#include "sixt_main.c"

int phoimg_main() {
  // Register HEAdas task
  set_toolname("phoimg");
  set_toolversion("0.5");

  try {
    // ---------- Initialization ----------
    healog(3) << "initialize ...\n";
    sixte::Parameters sim_params;
    sixte::initSixteRng(sim_params.seed);

    sixte::XMLData xml_data(sim_params.xml_file);
    sixte::Simulator simulator(sim_params, xml_data);

    // ---------- Simulation  ----------
    healog(3) << "\nstart simulation ...\n";
    simulator.runSimulation();

  } catch (const std::exception& e) {
    sixte::printError(e.what());
    sixt_destroy_rng();
    return EXIT_FAILURE;
  }

  // ---------- Clean up ----------
  healog(3) << "\ncleaning up ...\n";
  sixt_destroy_rng();

  healog(3) << "finished successfully!\n\n";
  return EXIT_SUCCESS;
}

sixte::Simulator::Simulator(const sixte::Parameters& sim_params, sixte::XMLData& xml_data)
    : gti_collection_(sim_params.gti_file, sim_params.time),
      obs_info_(xml_data, sim_params.pointing, gti_collection_),
      attitude_(sim_params.pointing, sim_params.time, sim_params.attitude_dt),
      photon_imaging_(sim_params.impact_list, sim_params.clobber,
                      xml_data, obs_info_),
      show_progress_(sim_params.show_progress),
      photonfile_(sim_params.photon_list){
}

void sixte::Simulator::runSimulation() {
  runObservation();
}

void sixte::Simulator::runObservation() {
  Progressbar progressbar(show_progress_, obs_info_.total_sim_time);

  sixte::Fitsfile infile(photonfile_, FileMode::read);
  infile.moveToExt("PHOTONS");

  long row = 1;
  std::optional<SixtePhoton> phot = getPhoton(infile, row);

  while (phot.has_value()) {
    for (const auto& gti_bin: gti_collection_.getAllGtiBins()) {
      if (phot->time() <= gti_bin.second) {
        photon_imaging_.doImaging(attitude_, phot.value());
        progressbar.update(phot->time());

        row++;
        phot = getPhoton(infile, row);

        if (!phot.has_value()) break;
      }
    }
  }

  progressbar.finish();
}

namespace sixte {
std::optional<SixtePhoton> getPhoton(Fitsfile& infile, size_t row) {

  if (row > infile.getNumRows()) {
    return std::nullopt;
  }

  double time, ra, dec;
  float energy;
  int ph_id, src_id;

  infile.readCol("TIME", row, time);
  infile.readCol("ENERGY", row, energy);
  infile.readCol("RA", row, ra);
  infile.readCol("DEC", row, dec);
  infile.readCol("PH_ID", row, ph_id);
  infile.readCol("SRC_ID", row, src_id);

  PhotonMetainfo meta(ph_id, src_id);
  SixtePhoton phot(time, energy, std::make_pair(ra*M_PI/180.,dec*M_PI/180.), meta);

  return phot;
}
}
