/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.

   Copyright 2020 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "phogen.hpp"

#define TOOLSUB phogen_main
#include "sixt_main.c"

int phogen_main() {
  // Register HEAdas task
  set_toolname("phogen");
  set_toolversion("0.5");

  try {
    // ---------- Initialization ----------
    healog(3) << "initialize ...\n";
    sixte::Parameters sim_params;
    sixte::initSixteRng(sim_params.seed);

    std::vector<sixte::XMLData> xml_data;
    xml_data.emplace_back((sim_params.xml_file));

    sixte::GTICollection gtis(sim_params.gti_file, sim_params.time);

    std::vector<sixte::ObsInfo> obs_info;
    obs_info.emplace_back(xml_data[0], sim_params.pointing, gtis);

    sixte::Simulator simulator(sim_params, xml_data, gtis, obs_info);

    // ---------- Simulation  ----------
    healog(3) << "\nstart simulation ...\n";
    simulator.runSimulation();

  } catch (const std::exception& e) {
    sixte::printError(e.what());
    sixt_destroy_rng();
    return EXIT_FAILURE;
  }

  // ---------- Clean up ----------
  healog(3) << "\ncleaning up ...\n";
  sixt_destroy_rng();

  healog(3) << "finished successfully!\n\n";
  return EXIT_SUCCESS;
}

sixte::Simulator::Simulator(const Parameters& sim_params, std::vector<XMLData>& xml_data, const GTICollection& gtis, const std::vector<ObsInfo>& obs_infos)
    : gti_collection_(gtis),
      attitude_(sim_params.pointing, sim_params.time, sim_params.attitude_dt),
      photon_generation_(sim_params.photon_list, sim_params.clobber,
                         sim_params.simput_files,
                         xml_data, obs_infos),
      simtime(obs_infos[0].total_sim_time),
      show_progress_(sim_params.show_progress) {
}

void sixte::Simulator::runSimulation() {
  runObservation();
}

void sixte::Simulator::runObservation() {
  Progressbar progressbar(show_progress_, simtime);

  for (const auto& gti_bin: gti_collection_.getAllGtiBins()) {
    std::optional<SixtePhoton> phot = std::nullopt;
    do {
      phot = photon_generation_.getNextPhoton(attitude_, gti_bin);
      if (phot.has_value()) {
        progressbar.update(phot->time());
      }
    } while (phot.has_value());
  }

  progressbar.finish();
}
