/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "makelc.hpp"
#include "Parameters.h"
#include "sixt.h"

#define TOOLSUB makelc_main
#include "sixt_main.c"


ClinePars read_par() {

  ClinePars pars;

  pars.EvtFile = sixte::queryParameterString("EvtFile");
  pars.LightCurve = sixte::queryParameterString("LightCurve");
  pars.GTIfile = sixte::queryParameterString("GTIfile");
  pars.regfile = sixte::queryParameterString("regfile");

  pars.tstart = sixte::queryParameterDouble("TSTART");
  pars.dt = sixte::queryParameterDouble("dt");
  pars.length = sixte::queryParameterDouble("length");
  pars.Emin = sixte::queryParameterDouble("Emin");
  pars.Emax = sixte::queryParameterDouble("Emax");
  pars.Chanmin = sixte::queryParameterInt("Chanmin");
  pars.Chanmax = sixte::queryParameterInt("Chanmax");
  pars.Chantype = sixte::queryParameterString("Chantype");


  pars.clobber = sixte::queryParameterBool("clobber");

  return pars;
}

void getHeaderVals(HeaderVals &vals,
    CCfits::ExtHDU &hdu) {

  // read a bunch of keywords
  try {
    hdu.readKey("TELESCOP", vals.telescop);
    hdu.readKey("INSTRUME", vals.instrume);
    hdu.readKey("FILTER", vals.filter);
    hdu.readKey("MJDREF", vals.mjdref);
    hdu.readKey("TIMEZERO", vals.timezero);
  } catch (CCfits::HDU::NoSuchKeyword &e) {
    throw sixte::SixteException("Missing keyword in EvtFile");
  }
}


// Note: The logic here is actually different from makespec
int getSigCol(CCfits::FITS &evtfile, std::string &Chantype, bool filter_en) {
  int colidx = -1;

  if (filter_en) {
    try {
        colidx = evtfile.currentExtension().column("SIGNAL").index();
        healog(5) << "Reading energies from column SIGNAL\n";
      } catch (CCfits::Table::NoSuchColumn){
        throw sixte::SixteException(
            "No SIGNAL column present in event file, even though energy filtering was requested");
      }
  } else {
    if (sixte::isNone(Chantype)) {
      throw sixte::SixteException(
          "No channel type specified for filtering. Please use the Chantype parameter");
    }
    if (Chantype.compare("PI") == 0) {
      try {
        colidx = evtfile.currentExtension().column("PI").index();
        healog(5) << "Reading energies from column PI\n";
      } catch (CCfits::Table::NoSuchColumn) {
      }
    }
    else if (Chantype.compare("PHA") == 0) {
      try {
        colidx = evtfile.currentExtension().column("PHA").index();
        healog(5) << "Reading energies from column PHA\n";
      } catch (CCfits::Table::NoSuchColumn) {
      }
    } else {
    }
  }

  if (colidx == -1) {
    throw sixte::SixteException(
        "Unable to locate a column specifying event energy in EvtFile (must be PHA, SIGNAL, or PI), even though energy filtering was requested");
  }

  return colidx;
}

int makelc_main() {
  // Register HEAdas task
  set_toolname("makelc");
  set_toolversion("1.0");

  // declare some NULL pointers to free later
  SAORegion* region = NULL;
  int status = EXIT_SUCCESS;

  try {
    ClinePars pars = read_par();

    CCfits::FITS evtfile(pars.EvtFile, CCfits::Read);
    evtfile.extension("EVENTS");

    // read values from the event file header
    HeaderVals headvals;
    getHeaderVals(headvals, evtfile.currentExtension());

    // get the GTIs, either a user specified file or header keywords
    sixte::ObsTime obs_time(headvals.mjdref, pars.tstart, pars.length);
    sixte::GTICollection gtis(pars.GTIfile, obs_time);

    // prepare region filtering
    if (!sixte::isNone(pars.regfile)) {
      region = sixte::getRegion(pars.regfile, evtfile.currentExtension());
    }

    // prepare energy filtering
    bool filter_en = (pars.Emin >= 0. && pars.Emax >= 0. );
    bool filter_chan = (pars.Chanmin >= 0 && pars.Chanmax >= 0 );

    if (filter_en && filter_chan) {
      throw sixte::SixteException("Requested both Energy and Channel filtering - we cannot do both at once");
    }

    // determine the column containing the measured energies
    // this is either PI, PHA or SIGNAL
    std::optional<int> sigcolidx = std::nullopt;
    if (filter_en || filter_chan) {
      sigcolidx = getSigCol(evtfile, pars.Chantype, filter_en);
    }


    // then, the actual binning

    healog(3) << "calculate light curve(s) ..." << std::endl;

    auto gtibins = gtis.getAllGtiBins();
    std::vector<std::vector<long>> lightcurves;

    // create some empty light curves
    for (auto &bin: gtibins) {
      double len = bin.second - bin.first;
      lightcurves.emplace_back(long(ceil(len/pars.dt)), 0);
    }

    // read relevant data
    auto nrows = evtfile.currentExtension().rows();

    std::vector<double> times;
    evtfile.currentExtension().column("TIME").read(times, 1, nrows);

    std::vector<long> x;
    std::vector<long> y;

    if (region != NULL) {
      evtfile.currentExtension().column("X").read(x, 1, nrows);
      evtfile.currentExtension().column("Y").read(y, 1, nrows);
    }


    std::vector<long> chans;
    std::vector<double> signals;

    if (sigcolidx.has_value()) {
      if (filter_en) {
        evtfile.currentExtension().column(sigcolidx.value()).read(signals, 1, nrows);
      } else {
        evtfile.currentExtension().column(sigcolidx.value()).read(chans, 1, nrows);
      }
    }

    for (size_t ii=0; ii<(size_t) nrows; ii++) {
      // are we in one of the GTIs?
      double evt_time = times[ii];
      auto found_gti = std::find_if(gtibins.begin(), gtibins.end(),
          [evt_time](auto p){return p.first <= evt_time && evt_time <= evt_time;}
          );
      if (found_gti == gtibins.end()) {
        continue;
      }
      size_t gti_idx = std::distance(gtibins.begin(), found_gti);

      // apply other filters
      bool region_valid = (region!=NULL) ? sixte::filterReg(region, x[ii],y[ii]) : true;
      bool signal_valid = true;
      if (filter_en) {
        signal_valid = pars.Emin <= signals[ii] && pars.Emax >= signals[ii];
      }
      if (filter_chan) {
        signal_valid = pars.Chanmin <= chans[ii] && pars.Chanmax >= chans[ii];
      }

      if (region_valid && signal_valid) {
        size_t lc_idx = (evt_time-gtibins[gti_idx].first)/pars.dt;
        lightcurves[gti_idx][lc_idx] += 1;
      }
    }

    healog(3) << "store light curve(s) ..." << std::endl;

    for (size_t i_lc=0; i_lc < lightcurves.size(); i_lc++) {
      std::string filename;
      if (lightcurves.size() == 1) {
        filename = pars.LightCurve;
      } else {
        // split up the filename before the extension
        size_t lastdot = pars.LightCurve.rfind('.');
        std::stringstream ss;
        ss << pars.LightCurve.substr(0,lastdot)
          << "_" << i_lc
          << pars.LightCurve.substr(lastdot, pars.LightCurve.length()-lastdot);
        filename = ss.str();
      }

      if (pars.clobber) filename = "!"+filename;
      CCfits::FITS outfile(filename, CCfits::Write);

      std::vector<std::string> ttype{"COUNTS"};
      std::vector<std::string> tform{"J"};
      std::vector<std::string> tunit{"counts"};

      auto table = outfile.addTable("COUNTS", lightcurves[i_lc].size(),ttype,tform,tunit);

      // write data
      table->column("COUNTS").write(lightcurves[i_lc], 1);

      // write header
      table->addKey("ORIGIN", "ECAP", "Origin of FITS File");
      table->addKey("CREATOR", "makelc", "Program that created this FITS file");
      table->addKey("TIMVERSN", "OGIP/93-003", "OGIP memo number for file format");
      table->addKey("TELESCOP", headvals.telescop, "");
      table->addKey("INSTRUME", headvals.instrume, "");
      table->addKey("FILTER", headvals.filter, "");
      table->addKey("TIMEUNIT", "s", "time unit");
      table->addKey("TIMEDEL", pars.dt, "time resolution");
      table->addKey("MJDREF", gtis.mjdref(), "reference MJD");
      table->addKey("TIMEZERO", headvals.timezero, "time offset");
      table->addKey("TIMEPIXR", 0., "time stamp at beginning of bin");
      table->addKey("TSTART", gtibins[i_lc].first, "start time");
      table->addKey("TSTOP", gtibins[i_lc].second, "stop time");

      if (region != NULL) {
        table->addKey("REGFILE", pars.regfile, "Region filter file", true);
      }

      if (filter_en) {
        table->addKey("E_MIN", pars.Emin, "low energy for channel (keV)");
        table->addKey("E_MAX", pars.Emax, "high energy for channel (keV)");
      }
      if (filter_en) {
        table->addKey("CHAN_MIN", pars.Chanmin, "low channel");
        table->addKey("CHAN_MAX", pars.Chanmax, "high channel");
      }


      // save input parameters
      HDpar_stamp(outfile.fitsPointer(), 1, &status);
      sixte::checkStatusThrow(status, "Failed doing par_stamp");
    }

  } catch (sixte::SixteException &e) {
    std::cout << e.what() << std::endl;
    status=EXIT_FAILURE;
  } catch (CCfits::FitsException &e) {
    std::cout << e.message() << std::endl;
    status=EXIT_FAILURE;
  }

  // cleanup
  if (region!=NULL) {
    fits_free_region(region);
  }
  sixt_destroy_rng();
  return status;
}
