/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.

   Copyright 2020 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "gendetsim.hpp"

#define TOOLSUB gendetsim_main
#include "sixt_main.c"

int gendetsim_main() {
  // Register HEAdas task
  set_toolname("gendetsim");
  set_toolversion("0.5");

  try {
    // ---------- Initialization ----------
    healog(3) << "initialize ...\n";
    sixte::Parameters sim_params;
    sixte::initSixteRng(sim_params.seed);

    sixte::XMLData xml_data(sim_params.xml_file);

    // need to override eventfile in case of microcalorimeter
    auto type = xml_data.child("detector").attributeAsString("type");
    if (type == "microcal") {
      std::copy(sim_params.output_files.raw_datas.begin(),
          sim_params.output_files.raw_datas.end(),
          sim_params.output_files.evt_files.begin());
    }

    sixte::Simulator simulator(sim_params, xml_data);

    // ---------- Simulation  ----------
    healog(3) << "\nstart simulation ...\n";
    simulator.runSimulation();

    simulator.gti_collection_.saveGtiExtension(
        CCfits::FITS(sim_params.output_files.raw_datas[0], CCfits::Write).fitsPointer()
        );
  } catch (const std::exception& e) {
    sixte::printError(e.what());
    sixt_destroy_rng();
    return EXIT_FAILURE;
  }

  // ---------- Clean up ----------
  healog(3) << "\ncleaning up ...\n";
  sixt_destroy_rng();

  healog(3) << "finished successfully!\n\n";
  return EXIT_SUCCESS;
}

sixte::Simulator::Simulator(const sixte::Parameters& sim_params, sixte::XMLData& xml_data)
    : gti_collection_(sim_params.gti_file, sim_params.time),
      obs_info_(xml_data, sim_params.pointing, gti_collection_),
      detector_(xml_data, sim_params.output_files, obs_info_,
          sim_params.skip_invalids, 0, 0, std::make_shared<sixte::RmfRegistry>()),
      impfile(sim_params.impact_file, FileMode::read),
      show_progress_(sim_params.show_progress){

  impfile.moveToExt(1);
  if (sim_params.background) {
    // TODO check if pha background in XML
    pha_background_.emplace(xml_data,
                            detector_.sensorGeometry(),
                            detector_.canonicalEbounds(),
                            0, 0);
  }

  if (sim_params.calibration_source) {
    calibration_source_ = ModulatedXraySource(xml_data, detector_.sensorGeometry());
  }
}

void sixte::Simulator::runSimulation() {
  runObservation();
}

void sixte::Simulator::runObservation() {
  Progressbar progressbar(show_progress_, obs_info_.total_sim_time);
  double simulated_time = 0.;

  for (const auto& gti_bin: gti_collection_.getAllGtiBins()) {
    sixte::Carriers carrier_buffer;

    if (auto src_photon_carrier = getNextSrcPhotonCarrier(gti_bin)) {
      carrier_buffer.push(src_photon_carrier);
    }

    if (pha_background_) {
      if (auto bkg_carrier = pha_background_->getNextBkgCarrier(gti_bin)) {
        carrier_buffer.push(bkg_carrier);
      }
    }

    if (calibration_source_) {
      if (auto cal_src_carrier = getNextCalibrationSourceCarrier(gti_bin)) {
        carrier_buffer.push(cal_src_carrier);
      }
    }

    for (;;) {
      if (carrier_buffer.size() > 3) {
        throw SixteException("More than three events in carrier buffer");
      }

      if (carrier_buffer.empty()) {
        break;
      }

      auto carrier = sixte::pop_next_carrier(carrier_buffer);

      detector_.propagateReadout(carrier->creationTime());
      detector_.addCarrier(*carrier);

      progressbar.update(carrier->creationTime() - gti_bin.first + simulated_time);

      if ( auto next_carrier =
           getNextCarrierOfType((SrcType) carrier->photon_metainfo().src_id_, gti_bin) ) {
        carrier_buffer.push(next_carrier);
      }
    }

    assert(carrier_buffer.empty());
    detector_.propagateReadout(gti_bin.second);
    simulated_time += gti_bin.second-gti_bin.first;
  }
  detector_.finishSimulation(gti_collection_.tstopLastGti());

  progressbar.finish();
}

sixte::CarrierPtr sixte::Simulator::getNextCarrierOfType(const SrcType& src_type,
                                                         std::pair<double, double> dt) {
  if (src_type > 0) {
    return sixte::Simulator::getNextSrcPhotonCarrier(dt);
  } else if (src_type == SrcType::pha_bkg) {
    return pha_background_->getNextBkgCarrier(dt);
  } else if (src_type == SrcType::aux_bkg) {
    throw SixteException("AUX type not implemented yet");
  } else if (src_type == SrcType::mxs) {
    return sixte::Simulator::getNextCalibrationSourceCarrier(dt);
  } else {
    throw SixteException("Unknown source type");
  }
}

sixte::CarrierPtr sixte::Simulator::getNextSrcPhotonCarrier(std::pair<double, double> dt) {
  static Carriers carrier_buffer{};

  // Return the next carrier. Otherwise, loop over photon generation, imaging,
  // and absorption, until new carriers are created.
  for (;;) {
    if (!carrier_buffer.empty()) return pop_next_carrier(carrier_buffer);

    auto src_photon_impact = getImpact(impfile, fitsrow);
    if (!src_photon_impact) {return nullptr;}; // no photons left
    if (src_photon_impact->time() < dt.first) {fitsrow++; continue;} // photon is in prev GTI
    if (src_photon_impact->time() > dt.second) {return nullptr;} // photon is in next GTI
    fitsrow++;

    auto src_photon_carriers = detector_.absorb(*src_photon_impact);

    transferCarriers(src_photon_carriers, carrier_buffer);
  }
}

sixte::CarrierPtr sixte::Simulator::getNextCalibrationSourceCarrier(std::pair<double, double> dt) {
  static Carriers carrier_buffer{};

  // Return the next carrier. Otherwise, loop over photon generation
  // and absorption, until new carriers are created.
  for (;;) {
    if (!carrier_buffer.empty()) return pop_next_carrier(carrier_buffer);

    auto cal_src_photon_impact = calibration_source_->getNextPhoton(dt);
    if (!cal_src_photon_impact) return nullptr;

    auto cal_src_photon_carriers = detector_.absorb(*cal_src_photon_impact);

    transferCarriers(cal_src_photon_carriers, carrier_buffer);
  }
}

namespace sixte {
std::optional<SixtePhoton> getImpact(sixte::Fitsfile& infile, size_t row) {

  if (row > infile.getNumRows()) {
    return std::nullopt;
  }

  double time, x, y;
  float energy;
  int ph_id, src_id;

  infile.readCol("TIME", row, time);
  infile.readCol("ENERGY", row, energy);
  infile.readCol("X", row, x);
  infile.readCol("Y", row, y);
  infile.readCol("PH_ID", row, ph_id);
  infile.readCol("SRC_ID", row, src_id);

  PhotonMetainfo meta(ph_id, src_id);
  SixtePoint pt(x, y, 0);
  SixtePhoton phot(time, energy, pt, meta);

  return phot;
}
}
