/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "XMLData.h"
#include "Parameters.h"

using namespace sixte;

TEST_CASE("XMLData Test", "[xml_data]") {
  std::vector<std::string> xml_filename = {"./data/instruments/athena-wfi/ld_wfi_ff_large_all_chips.xml"};
  auto xml_datas = detectorXMLs(xml_filename).xml_documents;
  
  XMLData xml_data(xml_datas[0], "./data/instruments/athena-wfi/");
  std::string test_filename = "./my_test.xml";
  xml_data.saveFile(test_filename);
  
  SECTION("get element attribute value as double") {
    /** function to be tested: */
    double test_sample = xml_data.child("detector").child("wcs").attributeAsDouble("xrpix");
    
    /** test: */
    double sample_solution = 256.5;
    REQUIRE_THAT(test_sample, Catch::Matchers::WithinAbs(sample_solution, 1e-9));
  }
  
  SECTION("get element attribute value as string") {
    /** function to be tested: */
    std::string test_sample = xml_data.child("telescope").child("psf").attributeAsString("filename");
    
    /** test: */
    std::string sample_solution = "athena_psf_15row.fits";
    REQUIRE(test_sample == sample_solution);
  }
  
  SECTION("get element attribute value as int") {
    /** function to be tested: */
    int test_sample = xml_data.child("detector").child("dimensions").attributeAsInt("xwidth");
    
    /** test: */
    int sample_solution = 512;
    REQUIRE(test_sample == sample_solution);
  }
  
  SECTION("contains Element") {
    /** function to be tested: */
    bool test_sample = xml_data.child("detector").hasChild("wcs");
    
    /** test: */
    bool sample_solution = true;
    REQUIRE(test_sample == sample_solution);
  }
  
  SECTION("contains Element Attribute") {
    /** function to be tested: */
    bool test_sample = xml_data.child("detector").child("wcs").hasAttribute("xrpix");
    
    /** test: */
    bool sample_solution = true;
    REQUIRE(test_sample == sample_solution);
  }
  
  SECTION("get Dirname") {
    /** function to be tested: */
    std::string test_sample = xml_data.dirname();

    /** test: */
    std::string sample_solution = "./data/instruments/athena-wfi/";
    REQUIRE(test_sample == sample_solution);
  }
  
  SECTION("clean-up") {
    if (std::remove(test_filename.c_str()) != 0) {
      throw SixteException("Failed to remove output xmlfile");
    }
  }
}
