/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "NewVignetting.h"
#include "NewSIXT.h"

using namespace sixte;

TEST_CASE("Vignetting Test", "[vignetting]") {

  std::string vig_filename = "data/instruments/dummy/dummy_vig_athena_20231120.fits";
  NewVignetting test_vig(vig_filename);

  SECTION("Constructor"){
    std::vector<double> test_energy{0.1375, 0.475, 1.2125, 2.5, 4.375, 7, 14.25};
    std::vector<double> test_theta{0, 0.008333334, 0.01666667, 0.03333334, 0.05, 0.08333334,
                                   0.1166667, 0.1666667, 0.25, 0.3333333, 0.4666667};
    for (auto& theta : test_theta) theta *= M_PI / 180.;
    std::vector<double> test_phi{0};
    for (auto& phi : test_phi) phi *= M_PI / 180.;

    for (size_t ii=0; ii<test_energy.size(); ii++) {
      REQUIRE_THAT(test_vig.energy()[ii], Catch::Matchers::WithinAbs(test_energy[ii], 1e-8));
    }
    for (size_t ii=0; ii<test_theta.size(); ii++) {
      REQUIRE_THAT(test_vig.theta()[ii], Catch::Matchers::WithinAbs(test_theta[ii], 1e-8));
    }
    for (size_t ii=0; ii<test_phi.size(); ii++) {
      REQUIRE_THAT(test_vig.phi()[ii], Catch::Matchers::WithinAbs(test_phi[ii], 1e-8));
    }
  }

  SECTION("Vignetting Array") {
    std::vector<double> test_vignetting_input{1, 0.9950600266, 0.9905499816, 0.9773499966, 0.9582800269, 0.9013800025,
                                              0.8481299877, 0.7613300085, 0.623480022, 0.5019599795, 0.3435899913, 1,
                                              0.9952300191, 0.9907600284, 0.977609992, 0.9582899809, 0.9014999866,
                                              0.8478800058, 0.7604799867, 0.622699976, 0.5007200241, 0.3424699903, 1,
                                              0.9952700138, 0.9921299815, 0.9778400064, 0.9590799809, 0.9033899903,
                                              0.8528299928, 0.7655199766, 0.6290900111, 0.5067700148, 0.3478499949, 1,
                                              0.9937999845, 0.9928900003, 0.9690600038, 0.9381399751, 0.8586300015,
                                              0.7832599878, 0.6604599953, 0.4844399989, 0.3572899997, 0.2182500064, 1,
                                              0.9967799783, 0.9947999716, 0.9677500129, 0.9336900115, 0.8435400128,
                                              0.753000021, 0.6218000054, 0.4195599854, 0.2816599905, 0.1551499963, 1,
                                              0.997609973, 0.9904999733, 0.9523599744, 0.8844299912, 0.7497500181,
                                              0.6239799857, 0.4374699891, 0.2348500043, 0.1441099942, 0.07350300252, 1,
                                              0.9836300015, 0.9743199944, 0.9050700068, 0.8258600235, 0.622870028,
                                              0.4610500038, 0.2842600048, 0.1468700022, 0.08833599836, 0.02992399968};
    std::vector<std::vector<std::vector<double>>> test_vignetting_array;
    size_t nenergy = test_vig.energy().size();
    size_t ntheta = test_vig.theta().size();
    size_t nphi = test_vig.phi().size();
    
    typedef std::vector<double> d1;
    typedef std::vector<d1> d2;
    typedef std::vector<d2> d3;
    test_vignetting_array = d3(nenergy, d2(ntheta, d1(nphi)));
    
    size_t count = 0;
    for(size_t ecount=0; ecount<nenergy; ecount++){
      for(size_t tcount=0; tcount<ntheta; tcount++){
        for(size_t pcount=0; pcount<nphi; pcount++){
          test_vignetting_array[ecount][tcount][pcount] = test_vignetting_input[count];
          count++;
        }
      }
    }

    for(size_t ecount=0; ecount<nenergy; ecount++){
      for(size_t tcount=0; tcount<ntheta; tcount++){
        for(size_t pcount=0; pcount<nphi; pcount++){
          REQUIRE_THAT(test_vig.vignetting_array()[ecount][tcount][pcount],
                       Catch::Matchers::WithinAbs(test_vignetting_array[ecount][tcount][pcount], 1e-9));
        }
      }
    }
  }

  SECTION("getVignettingFactor") {
    std::vector<double> test_factor_energy{1, 2.5, 4, 7, 12.1};
    std::vector<double> test_factor_theta{0.02, 0.05, 0.156, 0.25, 0.5};
    for (auto& theta : test_factor_theta) theta *= M_PI / 180.;
    std::vector<double> test_result{0.9889429808, 0.9381399751, 0.6571629047, 0.2348500043, 0.04284742475};

    for (size_t ii=0; ii<test_factor_energy.size(); ii++) {
      double result = test_vig.getVignettingFactor(test_factor_energy[ii], test_factor_theta[ii], 0.0);
      REQUIRE_THAT(result, Catch::Matchers::WithinAbs(test_result[ii], 1e-7));
    }
  }

}
