/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "SixteVector.h"


/*
 * TEST VALUES:
 *
 *
 */

using namespace sixte;

TEST_CASE("Sixte Vector", "[sixte_vector]") {
  double x = 4.7;
  double y = 7.4;
  double z = 7.474;
  Vector_3 vec(4.7, 7.4, 7.474);
  
  SixteVector sixte_vector_1(x, y, z);
  SixteVector sixte_vector_2;
  SixteVector sixte_vector_3(vec);
  
  SECTION("Unit Vector") {
    double ra = 56.98;
    double dec = 47.895;

    SixteVector test_output = unitVector(ra, dec);
    SixteVector test_results(-0.6514544619, -0.2998217918, -0.6969318311);

    REQUIRE_THAT(test_output.x(), Catch::Matchers::WithinAbs(test_results.x(), 1e-9));
    REQUIRE_THAT(test_output.y(), Catch::Matchers::WithinAbs(test_results.y(), 1e-9));
    REQUIRE_THAT(test_output.z(), Catch::Matchers::WithinAbs(test_results.z(), 1e-9));
  }
  
  SECTION("Normalized Vector") {
    SixteVector test_vec(x, y, z);
    test_vec = normalizeVector(test_vec);
    SixteVector test_results(0.4079856869, 0.6423604431, 0.6487840476);
    
    REQUIRE_THAT(test_vec.x(), Catch::Matchers::WithinAbs(test_results.x(), 1e-9));
    REQUIRE_THAT(test_vec.y(), Catch::Matchers::WithinAbs(test_results.y(), 1e-9));
    REQUIRE_THAT(test_vec.z(), Catch::Matchers::WithinAbs(test_results.z(), 1e-9));
  }
  
  SECTION("Scalar Product") {
    SixteVector test_vec1(x, y, z);
    SixteVector test_vec2(x+4, y+4, z+4);

    double test_output = scalarProduct(test_vec1, test_vec2);
    double test_result = 211.006676;

    REQUIRE_THAT(test_output, Catch::Matchers::WithinAbs(test_result, 1e-9));
  }

  SECTION("Cross Product") {
    /** Formally:vector_product */
    SixteVector test_vec1(x, y, z);
    SixteVector test_vec2(x+4, y+4, z+4);

    SixteVector test_output = crossProduct(test_vec1, test_vec2);
    SixteVector test_results(-0.296, 11.096, -10.8);

    REQUIRE_THAT(test_output.x(), Catch::Matchers::WithinAbs(test_results.x(), 1e-9));
    REQUIRE_THAT(test_output.y(), Catch::Matchers::WithinAbs(test_results.y(), 1e-9));
    REQUIRE_THAT(test_output.z(), Catch::Matchers::WithinAbs(test_results.z(), 1e-9));
  }

  SECTION("Vector Difference") {
    SixteVector test_vec1(x, y, z);
    SixteVector test_vec2(x+4, y+4, z+4);

    SixteVector test_output = vectorDifference(test_vec1, test_vec2);
    SixteVector test_results(-4, -4, -4);

    REQUIRE_THAT(test_output.x(), Catch::Matchers::WithinAbs(test_results.x(), 1e-9));
    REQUIRE_THAT(test_output.y(), Catch::Matchers::WithinAbs(test_results.y(), 1e-9));
    REQUIRE_THAT(test_output.z(), Catch::Matchers::WithinAbs(test_results.z(), 1e-9));
  }

  SECTION("Interpolate Vector") {
    SixteVector test_vec1(x, y, z);
    SixteVector test_vec2(x+4, y+4, z+4);
    double t1 = 39.57298;
    double t2 = 42.79482;
    double time = 40.4;

    SixteVector test_output = interpolateVec(test_vec1, t1, test_vec2, t2, time);
    SixteVector test_results(5.726767313, 8.426767313, 8.500767313);

    REQUIRE_THAT(test_output.x(), Catch::Matchers::WithinAbs(test_results.x(), 1e-9));
    REQUIRE_THAT(test_output.y(), Catch::Matchers::WithinAbs(test_results.y(), 1e-9));
    REQUIRE_THAT(test_output.z(), Catch::Matchers::WithinAbs(test_results.z(), 1e-9));
  }

  SECTION("Interpolate Circle Vector") {
    SixteVector test_vec1(x, y, z);
    SixteVector test_vec2(x+4, y+4, z+4);
    double phase = 0.727189;

    SixteVector test_output = interpolateCircleVector(test_vec1, test_vec2, phase);
    SixteVector test_results(0.4560146504,0.6269585021,0.6316436299);

    REQUIRE_THAT(test_output.x(), Catch::Matchers::WithinAbs(test_results.x(), 1e-9));
    REQUIRE_THAT(test_output.y(), Catch::Matchers::WithinAbs(test_results.y(), 1e-9));
    REQUIRE_THAT(test_output.z(), Catch::Matchers::WithinAbs(test_results.z(), 1e-9));
  }

  SECTION("Calculate Ra and Dec") {
    SixteVector test_vec1(x, y, z);
    std::pair<double, double> test_ra_dec = calculateRaDec(test_vec1);
    std::pair<double, double> test_results = std::make_pair(1.004942, 0.7059854536);

    REQUIRE_THAT(test_ra_dec.first, Catch::Matchers::WithinAbs(test_results.first, 1e-9));
    REQUIRE_THAT(test_ra_dec.second, Catch::Matchers::WithinAbs(test_results.second, 1e-9));
  }
}