/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "SixtePhoton.h"

using namespace sixte;

TEST_CASE("Sixte Photon Test", "[sixte_photon]") {
  /** test values */
  double time = 3.2;
  double energy = 38.13;

  double ra = 4.43232;
  double dec = 234.654;
  std::pair<double, double> sky_position(ra, dec);
  
  double pos_x = 9.0;
  double pos_y = 32.2;
  double pos_z = 4341.432;
  SixtePoint detector_position(pos_x, pos_y, pos_z);
  
  long ph_id = 23;
  long src_id = 2;
  size_t tel_id = 5;
  size_t chip_id = 3;
  PhotonMetainfo photon_metainfo(ph_id, src_id, tel_id, chip_id);
  
  Photon photon;
  photon.time = time;
  photon.energy = (float) energy;
  photon.ra = ra;
  photon.dec = dec;
  photon.ph_id = ph_id;
  photon.src_id = src_id;
  photon.c_tel_id = (int) tel_id;
  
  /** test constructors */
  SixtePhoton sixte_photon_1(time, energy, sky_position, detector_position, photon_metainfo);
  
  SixtePhoton sixte_photon_2(time, energy, sky_position, photon_metainfo);
  
  SixtePhoton sixte_photon_3(time, energy, detector_position, photon_metainfo);
  
  SixtePhoton sixte_photon_4(photon);
  
  SixtePhoton sixte_photon_5(photon, tel_id);
  
  SECTION("Time") {
    /** get test values */
    double test_time_1 = sixte_photon_1.time();
    double test_time_2 = sixte_photon_2.time();
    double test_time_3 = sixte_photon_3.time();
    double test_time_4 = sixte_photon_4.time();
    double test_time_5 = sixte_photon_5.time();
    
    /** test */
    REQUIRE_THAT(test_time_1, Catch::Matchers::WithinAbs(time, 1e-9));
    REQUIRE_THAT(test_time_2, Catch::Matchers::WithinAbs(time, 1e-9));
    REQUIRE_THAT(test_time_3, Catch::Matchers::WithinAbs(time, 1e-9));
    REQUIRE_THAT(test_time_4, Catch::Matchers::WithinAbs(time, 1e-9));
    REQUIRE_THAT(test_time_5, Catch::Matchers::WithinAbs(time, 1e-9));
  }
  
  SECTION("Energy") {
    /** get test values */
    double test_energy_1 = sixte_photon_1.energy();
    double test_energy_2 = sixte_photon_2.energy();
    double test_energy_3 = sixte_photon_3.energy();
    double test_energy_4 = sixte_photon_4.energy();
    double test_energy_5 = sixte_photon_5.energy();
    
    /** test */
    REQUIRE_THAT(test_energy_1, Catch::Matchers::WithinAbs(energy, 1e-9));
    REQUIRE_THAT(test_energy_2, Catch::Matchers::WithinAbs(energy, 1e-9));
    REQUIRE_THAT(test_energy_3, Catch::Matchers::WithinAbs(energy, 1e-9));
    REQUIRE_THAT(test_energy_4, Catch::Matchers::WithinAbs(energy, 1e-5));
    REQUIRE_THAT(test_energy_5, Catch::Matchers::WithinAbs(energy, 1e-5));
  }
  
  SECTION("Detector Position") {
    if (sixte_photon_1.detector_position().has_value()) {
      /** get test values */
      double test_detector_position_1x = sixte_photon_1.detector_position()->x();
      double test_detector_position_1y = sixte_photon_1.detector_position()->y();
      double test_detector_position_1z = sixte_photon_1.detector_position()->z();
      
      /** test */
      REQUIRE_THAT(test_detector_position_1x, Catch::Matchers::WithinAbs(detector_position.x(), 1e-9));
      REQUIRE_THAT(test_detector_position_1y, Catch::Matchers::WithinAbs(detector_position.y(), 1e-9));
      REQUIRE_THAT(test_detector_position_1z, Catch::Matchers::WithinAbs(detector_position.z(), 1e-9));
    }
    
    if (sixte_photon_2.detector_position().has_value()) {
      /** get test values */
      double test_detector_position_2x = sixte_photon_2.detector_position()->x();
      double test_detector_position_2y = sixte_photon_2.detector_position()->y();
      double test_detector_position_2z = sixte_photon_2.detector_position()->z();
      
      /** test */
      REQUIRE_THAT(test_detector_position_2x, Catch::Matchers::WithinAbs(detector_position.x(), 1e-9));
      REQUIRE_THAT(test_detector_position_2y, Catch::Matchers::WithinAbs(detector_position.y(), 1e-9));
      REQUIRE_THAT(test_detector_position_2z, Catch::Matchers::WithinAbs(detector_position.z(), 1e-9));
    }
    
    if (sixte_photon_3.detector_position().has_value()) {
      /** get test values */
      double test_detector_position_3x = sixte_photon_3.detector_position()->x();
      double test_detector_position_3y = sixte_photon_3.detector_position()->y();
      double test_detector_position_3z = sixte_photon_3.detector_position()->z();
      
      /** test */
      REQUIRE_THAT(test_detector_position_3x, Catch::Matchers::WithinAbs(detector_position.x(), 1e-9));
      REQUIRE_THAT(test_detector_position_3y, Catch::Matchers::WithinAbs(detector_position.y(), 1e-9));
      REQUIRE_THAT(test_detector_position_3z, Catch::Matchers::WithinAbs(detector_position.z(), 1e-9));
    }
    
    if (sixte_photon_4.detector_position().has_value()) {
      /** get test values */
      double test_detector_position_4x = sixte_photon_4.detector_position()->x();
      double test_detector_position_4y = sixte_photon_4.detector_position()->y();
      double test_detector_position_4z = sixte_photon_4.detector_position()->z();
      
      /** test */
      REQUIRE_THAT(test_detector_position_4x, Catch::Matchers::WithinAbs(detector_position.x(), 1e-9));
      REQUIRE_THAT(test_detector_position_4y, Catch::Matchers::WithinAbs(detector_position.y(), 1e-9));
      REQUIRE_THAT(test_detector_position_4z, Catch::Matchers::WithinAbs(detector_position.z(), 1e-9));
    }
    
    if (sixte_photon_5.detector_position().has_value()) {
      /** get test values */
      double test_detector_position_5x = sixte_photon_5.detector_position()->x();
      double test_detector_position_5y = sixte_photon_5.detector_position()->y();
      double test_detector_position_5z = sixte_photon_5.detector_position()->z();
      
      /** test */
      REQUIRE_THAT(test_detector_position_5x, Catch::Matchers::WithinAbs(detector_position.x(), 1e-9));
      REQUIRE_THAT(test_detector_position_5y, Catch::Matchers::WithinAbs(detector_position.y(), 1e-9));
      REQUIRE_THAT(test_detector_position_5z, Catch::Matchers::WithinAbs(detector_position.z(), 1e-9));
    }
  }
  
  SECTION("Photon Metainfo") {
    /** get test values */
    long test_ph_id_1 = sixte_photon_1.photon_metainfo().ph_id_;
    long test_src_id_1 = sixte_photon_1.photon_metainfo().src_id_;
    size_t test_tel_id_1 = sixte_photon_1.photon_metainfo().tel_id_;
    size_t test_chip_id_1 = sixte_photon_1.photon_metainfo().chip_id_;
    
    long test_ph_id_2 = sixte_photon_2.photon_metainfo().ph_id_;
    long test_src_id_2 = sixte_photon_2.photon_metainfo().src_id_;
    size_t test_tel_id_2 = sixte_photon_2.photon_metainfo().tel_id_;
    size_t test_chip_id_2 = sixte_photon_2.photon_metainfo().chip_id_;
    
    long test_ph_id_3 = sixte_photon_3.photon_metainfo().ph_id_;
    long test_src_id_3 = sixte_photon_3.photon_metainfo().src_id_;
    size_t test_tel_id_3 = sixte_photon_3.photon_metainfo().tel_id_;
    size_t test_chip_id_3 = sixte_photon_3.photon_metainfo().chip_id_;
    
    long test_ph_id_4 = sixte_photon_4.photon_metainfo().ph_id_;
    long test_src_id_4 = sixte_photon_4.photon_metainfo().src_id_;
    size_t test_tel_id_4 = sixte_photon_4.photon_metainfo().tel_id_;
    size_t test_chip_id_4 = sixte_photon_4.photon_metainfo().chip_id_;
    
    long test_ph_id_5 = sixte_photon_5.photon_metainfo().ph_id_;
    long test_src_id_5 = sixte_photon_5.photon_metainfo().src_id_;
    size_t test_tel_id_5 = sixte_photon_5.photon_metainfo().tel_id_;
    size_t test_chip_id_5 = sixte_photon_5.photon_metainfo().chip_id_;
    
    /** test */
    REQUIRE(test_ph_id_1 == photon_metainfo.ph_id_);
    REQUIRE(test_src_id_1 == photon_metainfo.src_id_);
    REQUIRE(test_tel_id_1 == photon_metainfo.tel_id_);
    REQUIRE(test_chip_id_1 == photon_metainfo.chip_id_);
    
    REQUIRE(test_ph_id_2 == photon_metainfo.ph_id_);
    REQUIRE(test_src_id_2 == photon_metainfo.src_id_);
    REQUIRE(test_tel_id_2 == photon_metainfo.tel_id_);
    REQUIRE(test_chip_id_2 == photon_metainfo.chip_id_);
    
    REQUIRE(test_ph_id_3 == photon_metainfo.ph_id_);
    REQUIRE(test_src_id_3 == photon_metainfo.src_id_);
    REQUIRE(test_tel_id_3 == photon_metainfo.tel_id_);
    REQUIRE(test_chip_id_3 == photon_metainfo.chip_id_);
    
    REQUIRE(test_ph_id_4 == photon_metainfo.ph_id_);
    REQUIRE(test_src_id_4 == photon_metainfo.src_id_);
    REQUIRE(test_tel_id_4 == 0);
    REQUIRE(test_chip_id_4 == 0);
    
    REQUIRE(test_ph_id_5 == photon_metainfo.ph_id_);
    REQUIRE(test_src_id_5 == photon_metainfo.src_id_);
    REQUIRE(test_tel_id_5 == photon_metainfo.tel_id_);
    REQUIRE(test_chip_id_5 == 0);
  }
  
  SECTION("Ra") {
    if (sixte_photon_1.ra().has_value()) {
      /** get test values */
      double test_ra_1 = sixte_photon_1.ra().value();
      /** test */
      REQUIRE_THAT(test_ra_1, Catch::Matchers::WithinAbs(sky_position.first, 1e-9));
    }
    
    if (sixte_photon_2.ra().has_value()) {
      /** get test values */
      double test_ra_2 = sixte_photon_2.ra().value();
      /** test */
      REQUIRE_THAT(test_ra_2, Catch::Matchers::WithinAbs(sky_position.first, 1e-9));
    }
    
    if (sixte_photon_3.ra().has_value()) {
      /** get test values */
      double test_ra_3 = sixte_photon_3.ra().value();
      /** test */
      REQUIRE_THAT(test_ra_3, Catch::Matchers::WithinAbs(sky_position.first, 1e-9));
    }
    
    if (sixte_photon_4.ra().has_value()) {
      /** get test values */
      double test_ra_4 = sixte_photon_4.ra().value();
      /** test */
      REQUIRE_THAT(test_ra_4, Catch::Matchers::WithinAbs(sky_position.first, 1e-9));
    }
    
    if (sixte_photon_5.ra().has_value()) {
      /** get test values */
      double test_ra_5 = sixte_photon_5.ra().value();
      /** test */
      REQUIRE_THAT(test_ra_5, Catch::Matchers::WithinAbs(sky_position.first, 1e-9));
    }
  }
  
  SECTION("Dec") {
    if (sixte_photon_1.dec().has_value()) {
      /** get test values */
      double test_dec_1 = sixte_photon_1.dec().value();
      /** test */
      REQUIRE_THAT(test_dec_1, Catch::Matchers::WithinAbs(sky_position.second, 1e-9));
    }
    
    if (sixte_photon_2.dec().has_value()) {
      /** get test values */
      double test_dec_2 = sixte_photon_2.dec().value();
      /** test */
      REQUIRE_THAT(test_dec_2, Catch::Matchers::WithinAbs(sky_position.second, 1e-9));
    }
    
    if (sixte_photon_3.dec().has_value()) {
      /** get test values */
      double test_dec_3 = sixte_photon_3.dec().value();
      /** test */
      REQUIRE_THAT(test_dec_3, Catch::Matchers::WithinAbs(sky_position.second, 1e-9));
    }
    
    if (sixte_photon_4.dec().has_value()) {
      /** get test values */
      double test_dec_4 = sixte_photon_4.dec().value();
      /** test */
      REQUIRE_THAT(test_dec_4, Catch::Matchers::WithinAbs(sky_position.second, 1e-9));
    }
    
    if (sixte_photon_5.dec().has_value()) {
      /** get test values */
      double test_dec_5 = sixte_photon_5.dec().value();
      /** test */
      REQUIRE_THAT(test_dec_5, Catch::Matchers::WithinAbs(sky_position.second, 1e-9));
    }
  }
  
  SECTION("C Photon") {
    /** get test values */
    auto test_c_photon_1 = sixte_photon_1.c_photon();
    auto test_c_photon_2 = sixte_photon_2.c_photon();
    auto test_c_photon_3 = sixte_photon_3.c_photon();
    auto test_c_photon_4 = sixte_photon_4.c_photon();
    auto test_c_photon_5 = sixte_photon_5.c_photon();
    
    /** test */
    REQUIRE_THAT(test_c_photon_1->time, Catch::Matchers::WithinAbs(time, 1e-9));
    REQUIRE_THAT(test_c_photon_1->energy, Catch::Matchers::WithinAbs(energy, 1e-5));
    REQUIRE_THAT(test_c_photon_1->ra, Catch::Matchers::WithinAbs(sky_position.first, 1e-9));
    REQUIRE_THAT(test_c_photon_1->dec, Catch::Matchers::WithinAbs(sky_position.second, 1e-9));
    REQUIRE(test_c_photon_1->ph_id == photon_metainfo.ph_id_);
    REQUIRE(test_c_photon_1->src_id == photon_metainfo.src_id_);
    REQUIRE(test_c_photon_1->c_tel_id == (int) photon_metainfo.tel_id_);
    
    REQUIRE_THAT(test_c_photon_2->time, Catch::Matchers::WithinAbs(time, 1e-9));
    REQUIRE_THAT(test_c_photon_2->energy, Catch::Matchers::WithinAbs(energy, 1e-5));
    REQUIRE_THAT(test_c_photon_2->ra, Catch::Matchers::WithinAbs(sky_position.first, 1e-9));
    REQUIRE_THAT(test_c_photon_2->dec, Catch::Matchers::WithinAbs(sky_position.second, 1e-9));
    REQUIRE(test_c_photon_2->ph_id == photon_metainfo.ph_id_);
    REQUIRE(test_c_photon_2->src_id == photon_metainfo.src_id_);
    REQUIRE(test_c_photon_2->c_tel_id == (int) photon_metainfo.tel_id_);
    
    REQUIRE_THAT(test_c_photon_3->time, Catch::Matchers::WithinAbs(time, 1e-9));
    REQUIRE_THAT(test_c_photon_3->energy, Catch::Matchers::WithinAbs(energy, 1e-5));
    REQUIRE_THAT(test_c_photon_3->ra, Catch::Matchers::WithinAbs(0, 1e-9));
    REQUIRE_THAT(test_c_photon_3->dec, Catch::Matchers::WithinAbs(0, 1e-9));
    REQUIRE(test_c_photon_3->ph_id == photon_metainfo.ph_id_);
    REQUIRE(test_c_photon_3->src_id == photon_metainfo.src_id_);
    REQUIRE(test_c_photon_3->c_tel_id == (int) photon_metainfo.tel_id_);
    
    REQUIRE_THAT(test_c_photon_4->time, Catch::Matchers::WithinAbs(time, 1e-9));
    REQUIRE_THAT(test_c_photon_4->energy, Catch::Matchers::WithinAbs(energy, 1e-5));
    REQUIRE_THAT(test_c_photon_4->ra, Catch::Matchers::WithinAbs(sky_position.first, 1e-9));
    REQUIRE_THAT(test_c_photon_4->dec, Catch::Matchers::WithinAbs(sky_position.second, 1e-9));
    REQUIRE(test_c_photon_4->ph_id == photon_metainfo.ph_id_);
    REQUIRE(test_c_photon_4->src_id == photon_metainfo.src_id_);
    REQUIRE(test_c_photon_4->c_tel_id == 0);
    
    REQUIRE_THAT(test_c_photon_5->time, Catch::Matchers::WithinAbs(time, 1e-9));
    REQUIRE_THAT(test_c_photon_5->energy, Catch::Matchers::WithinAbs(energy, 1e-5));
    REQUIRE_THAT(test_c_photon_5->ra, Catch::Matchers::WithinAbs(sky_position.first, 1e-9));
    REQUIRE_THAT(test_c_photon_5->dec, Catch::Matchers::WithinAbs(sky_position.second, 1e-9));
    REQUIRE(test_c_photon_5->ph_id == photon_metainfo.ph_id_);
    REQUIRE(test_c_photon_5->src_id == photon_metainfo.src_id_);
    REQUIRE(test_c_photon_5->c_tel_id == (int) photon_metainfo.tel_id_);
  }
  
  SECTION("Set Telescope ID") {
    /** test values */
    size_t set_tel_id = 4;

    /** function to be tested */
    sixte_photon_1.setTelID(set_tel_id);
    sixte_photon_2.setTelID(set_tel_id);
    sixte_photon_3.setTelID(set_tel_id);
    sixte_photon_4.setTelID(set_tel_id);
    sixte_photon_5.setTelID(set_tel_id);
    
    size_t test_tel_id_1 = sixte_photon_1.photon_metainfo().tel_id_;
    size_t test_tel_id_2 = sixte_photon_2.photon_metainfo().tel_id_;
    size_t test_tel_id_3 = sixte_photon_3.photon_metainfo().tel_id_;
    size_t test_tel_id_4 = sixte_photon_4.photon_metainfo().tel_id_;
    size_t test_tel_id_5 = sixte_photon_5.photon_metainfo().tel_id_;
    
    /** test */
    REQUIRE(test_tel_id_1 == set_tel_id);
    REQUIRE(test_tel_id_2 == set_tel_id);
    REQUIRE(test_tel_id_3 == set_tel_id);
    REQUIRE(test_tel_id_4 == set_tel_id);
    REQUIRE(test_tel_id_5 == set_tel_id);
  }
  
  SECTION("Set Chip ID") {
    /** test values */
    size_t set_chip_id = 4;
    
    /** function to be tested */
    sixte_photon_1.setChipID(set_chip_id);
    sixte_photon_2.setChipID(set_chip_id);
    sixte_photon_3.setChipID(set_chip_id);
    sixte_photon_4.setChipID(set_chip_id);
    sixte_photon_5.setChipID(set_chip_id);
    
    size_t test_chip_id_1 = sixte_photon_1.photon_metainfo().chip_id_;
    size_t test_chip_id_2 = sixte_photon_2.photon_metainfo().chip_id_;
    size_t test_chip_id_3 = sixte_photon_3.photon_metainfo().chip_id_;
    size_t test_chip_id_4 = sixte_photon_4.photon_metainfo().chip_id_;
    size_t test_chip_id_5 = sixte_photon_5.photon_metainfo().chip_id_;
    
    /** test */
    REQUIRE(test_chip_id_1 == set_chip_id);
    REQUIRE(test_chip_id_2 == set_chip_id);
    REQUIRE(test_chip_id_3 == set_chip_id);
    REQUIRE(test_chip_id_4 == set_chip_id);
    REQUIRE(test_chip_id_5 == set_chip_id);
  }

  SECTION("Pop Next Photon") {
    /** test values */
    double test_time_1 = 3.4;
    double test_time_2 = 0.12354;
    double test_time_3 = 894032;
    
    double test_energy = 678.9;
    SixtePoint test_detector_position(23.5, 2.0, 12.2);
    std::pair<double, double> test_sky_position(2, 578349);
    PhotonMetainfo test_photon_metainfo(6, 1, 4, 5);
    
    SixtePhoton sixte_test_photon_1(test_time_1, energy, sky_position, detector_position, photon_metainfo);
    SixtePhoton sixte_test_photon_2(test_time_2, energy, sky_position, detector_position, photon_metainfo);
    SixtePhoton sixte_test_photon_3(test_time_3, energy, sky_position, detector_position, photon_metainfo);
    SixtePhoton sixte_test_photon_4(test_time_1, test_energy, test_sky_position,
                                    test_detector_position, test_photon_metainfo);
    
    /** function to be tested */
    SixtePhotons photon_buffer;
    
    photon_buffer.emplace(sixte_test_photon_1);
    photon_buffer.emplace(sixte_test_photon_2);
    photon_buffer.emplace(sixte_test_photon_3);
    photon_buffer.emplace(sixte_test_photon_4);
    
    std::vector<SixtePhoton> photon_vec;
    
    while (!photon_buffer.empty()) photon_vec.emplace_back(pop_next_photon(photon_buffer));
    
    /** test */
    REQUIRE(photon_vec.size() == 4);
    REQUIRE_THAT(photon_vec[0].time(), Catch::Matchers::WithinAbs(test_time_2, 1e-9));
    REQUIRE_THAT(photon_vec[1].time(), Catch::Matchers::WithinAbs(test_time_1, 1e-9));
    REQUIRE_THAT(photon_vec[2].time(), Catch::Matchers::WithinAbs(test_time_1, 1e-9));
    REQUIRE_THAT(photon_vec[3].time(), Catch::Matchers::WithinAbs(test_time_3, 1e-9));
  }
}