/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "SixteCCFits.h"

using namespace sixte;

TEST_CASE("SIXTE CCFits Test", "[sixteccfits]") {
  std::string filename = "sixteccfits_test.fits";

  SECTION("Create dummy file from template") {
    auto fitsptr = EventFileTemplate("!" + filename);
  }

  SECTION("Fitsfile write functions") {
    Fitsfile test_fitsfile(filename, FileMode::write);
    test_fitsfile.moveToExt("EVENTS");

    // AddCol
    test_fitsfile.addCol(3, "SIXTE", "J", "ADU");

    // UpdateKey
    test_fitsfile.updateKey("TEST_INT", 7, "A test integer key");
    test_fitsfile.updateKey("TEST_STR", "Simulation of X-ray Telescopes", "A test string key");
    double test_double = 3.14;
    test_fitsfile.updateKey("TEST_DBL", test_double, "A test double key");

    // WriteCol
    test_fitsfile.writeCol("TIME", 1, 0.5);
    test_fitsfile.writeCol("SIGNAL", 1, 7.5);
    test_fitsfile.writeCol("SIXTE", 1, 100);

    double time = 1.5;
    test_fitsfile.writeCol("TIME", 2, time);
    int energy_int = 9;
    test_fitsfile.writeCol("SIGNAL", 2, energy_int); // Writing int to double column
    double sixte_double = 20.5;
    test_fitsfile.writeCol("SIXTE", 2, sixte_double); // Writing double to int column

    std::vector<double> ra_values{1.2, 4.9};
    test_fitsfile.writeCol("RA", 1, ra_values.size(), ra_values);

    // CreateImage
    test_fitsfile.createImage(ImageType::long_img, std::vector<long>{2, 3}, "TEST_IMAGE");

    // WritePix
    test_fitsfile.writePix(std::vector<long>{1, 1}, 6, std::vector<long>{1, 2, 3, 4, 5, 6});
  }

  SECTION("Fitsfile read functions") {
    Fitsfile test_fitsfile(filename, FileMode::write);
    test_fitsfile.moveToExt("EVENTS");

    // getColNum
    REQUIRE(test_fitsfile.getColNum("TIME") == 1);
    REQUIRE(test_fitsfile.getColNum("SIXTE") == 3);

    // getNumRows
    REQUIRE(test_fitsfile.getNumRows() == 2);

    // getNumCols
    REQUIRE(test_fitsfile.getNumCols() == 17);

    // readKey
    int test_int;
    test_fitsfile.readKey("TEST_INT", test_int);
    REQUIRE(test_int == 7);

    std::string test_str;
    test_fitsfile.readKey("TEST_STR", test_str);
    REQUIRE(test_str == "Simulation of X-ray Telescopes");

    double test_dbl;
    test_fitsfile.readKey("TEST_DBL", test_dbl);
    REQUIRE_THAT(test_dbl, Catch::Matchers::WithinAbs(3.14, 1e-9));

    // readCol
    double time;
    test_fitsfile.readCol("TIME", 1, time);
    REQUIRE_THAT(time, Catch::Matchers::WithinAbs(0.5, 1e-9));
    test_fitsfile.readCol("TIME", 2, time);
    REQUIRE_THAT(time, Catch::Matchers::WithinAbs(1.5, 1e-9));

    double signal;
    test_fitsfile.readCol("SIGNAL", 1, signal);
    REQUIRE_THAT(signal, Catch::Matchers::WithinAbs(7.5, 1e-9));
    test_fitsfile.readCol("SIGNAL", 2, signal);
    REQUIRE_THAT(signal, Catch::Matchers::WithinAbs(9, 1e-9));

    std::vector<int> sixte;
    test_fitsfile.readCol("SIXTE", 1, 2, sixte);
    REQUIRE(sixte[0] == 100);
    REQUIRE(sixte[1] == 20);

    std::vector<double> ra_values{1.1, 2.7, 3.9, 4.2}; // Initialize with random values
    test_fitsfile.readCol("RA", 1, 2, ra_values);
    REQUIRE_THAT(ra_values[0], Catch::Matchers::WithinAbs(1.2, 1e-9));
    REQUIRE_THAT(ra_values[1], Catch::Matchers::WithinAbs(4.9, 1e-9));

    // readPix
    test_fitsfile.moveToExt("TEST_IMAGE");
    std::vector<long> pix_values;
    test_fitsfile.readPix(std::vector<long>{1, 1}, 6, pix_values);
    REQUIRE(pix_values[0] == 1);
    REQUIRE(pix_values[1] == 2);
    REQUIRE(pix_values[2] == 3);
    REQUIRE(pix_values[3] == 4);
    REQUIRE(pix_values[4] == 5);
    REQUIRE(pix_values[5] == 6);
  }

  SECTION("clean-up") {
    if (std::remove(filename.c_str()) != 0) {
      throw SixteException("Failed to remove test fitsfile");
    }
  }
}

