/*
    This file is part of SIXTE.

    SIXTE is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    SIXTE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    For a copy of the GNU General Public License see
    <http://www.gnu.org/licenses/>.


    Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
    Erlangen-Nuernberg
*/


/*
#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "raytracing/Raytracing.h"
#include "raytracing/shape/Paraboloid.h"
#include "raytracing/geometry/Ray.h"

#include "sixte_random.h"

TEST_CASE("Ray-tracing Paraboloid", "[ray_tracing_paraboloid]") {
    Paraboloid_parameters paraboloidParameters{0.25, 0 ,0 ,10 ,20 ,0};
    Paraboloid paraboloid1(paraboloidParameters);
    SECTION("Intersection Easy") {
        sixte::SixteVector direction{0, 0, -1};
        Point_3 position{0, 0, 0};
        auto ray1 = Ray(direction, position, 1);
        Point_3 intersection_result{0, 0, -paraboloidParameters.p/2};
        Point_3 test_result = paraboloid1.get_intersection(ray1);
        REQUIRE_THAT(CGAL::to_double(intersection_result.x()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.x()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result.y()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.y()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result.z()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.z()), 1e-9));
    }

    SECTION("Intersection Easy Random") {
        sixte::initSixteRng(1);
        for (int ii = 0; ii < 100; ii ++) {
            double uniform_number = sixte::getUniformRandomNumber();
            double x = -20 + (20+20) * uniform_number;
            uniform_number = sixte::getUniformRandomNumber();
            double y = -20 + (20+20) * uniform_number;
            sixte::SixteVector direction{0, 0, -1};
            Point_3 position{x, y, 100};
            auto ray1 = Ray(direction, position, 1);
            Point_3 intersection_result{x, y, (x*x+y*y)/(2*paraboloidParameters.p)-paraboloidParameters.p/2};
            Point_3 test_result = paraboloid1.get_intersection(ray1);
            REQUIRE_THAT(CGAL::to_double(intersection_result.x()),
                         Catch::Matchers::WithinAbs(CGAL::to_double(test_result.x()), 1e-9));
            REQUIRE_THAT(CGAL::to_double(intersection_result.y()),
                         Catch::Matchers::WithinAbs(CGAL::to_double(test_result.y()), 1e-9));
            REQUIRE_THAT(CGAL::to_double(intersection_result.z()),
                         Catch::Matchers::WithinAbs(CGAL::to_double(test_result.z()), 1e-9));
        }
    }

    SECTION("Intersection from direction vector x, y != 0") {
        sixte::SixteVector direction{0.640703866726714, 1.065720402226051, -5.819999999999999};
            Point_3 position{0.96, -1.05, 10.82};
        auto ray1 = Ray(direction, position, 1);
        Point_3 intersection_result{1.600703866726714, 0.015720402226051, 5.000000000000001};
        Point_3 test_result = paraboloid1.get_intersection(ray1);
        REQUIRE_THAT(CGAL::to_double(intersection_result.x()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.x()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result.y()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.y()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result.z()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.z()), 1e-9));

        sixte::SixteVector direction_1{15.33532696758503, 1.154503356214399, -9429.385228070843};
        Point_3 position_1{-0.96,-10.05,10000.82};
        ray1 = Ray(direction_1, position_1, 1);
        Point_3 intersection_result_1{14.37532696758503, -8.895496643785602, 571.4347719291566};
        test_result = paraboloid1.get_intersection(ray1);
        REQUIRE_THAT(CGAL::to_double(intersection_result_1.x()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.x()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result_1.y()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.y()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result_1.z()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.z()), 1e-9));

        sixte::SixteVector direction_2{sixte::normalizeVector({3.85009100758503, 1.348046608519399, -9428.821582396426})};
        Point_3 position_2{10.52523596,-10.243543252305,10000.256354325582};
        ray1 = Ray(direction_2, position_2, 1);
        Point_3 intersection_result_2{14.37532696758503, -8.895496643785602, 571.4347719291566};
        test_result = paraboloid1.get_intersection(ray1);
        REQUIRE_THAT(CGAL::to_double(intersection_result_2.x()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.x()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result_2.y()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.y()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result_2.z()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.z()), 1e-9));
    }

    SECTION("Angle verification") {
        // Angle should be 90 deg
        sixte::SixteVector direction{0, 0, -1};
        Point_3 position{0, 0, 0};
        auto ray1 = Ray(direction, position, 1);
        double angle_result = -90.0;
        auto intersection = paraboloid1.get_intersection(ray1);
        auto normal = paraboloid1.get_normal_vector(intersection);
        double test_result = paraboloid1.get_angle(normal, direction);
        REQUIRE_THAT(angle_result,  Catch::Matchers::WithinAbs(test_result, 1e-9));

        // Angle should be 45 deg
        direction = {0, 0, -1};
        position = {0.25, 0, 10};
        ray1 = Ray(direction, position, 1);
        angle_result = -45.0;
        intersection = paraboloid1.get_intersection(ray1);
        normal = paraboloid1.get_normal_vector(intersection);
        test_result = paraboloid1.get_angle(normal, direction);
        REQUIRE_THAT(angle_result,  Catch::Matchers::WithinAbs(test_result, 1e-9));

        // Outwards of surface
        direction = {-1, -1, 0};
        position = {10, 10, 50};
        ray1 = Ray(direction, position, 1);
        intersection = paraboloid1.get_intersection(ray1);
        normal = paraboloid1.get_normal_vector(intersection);
        test_result = paraboloid1.get_angle(normal, direction);
        REQUIRE(test_result > 0);

        direction = {0., 0, 1};
        position = {0.25, 0, 0};
        ray1 = Ray(direction, position, 1);
        intersection = paraboloid1.get_intersection(ray1);
        normal = paraboloid1.get_normal_vector(intersection);
        test_result = paraboloid1.get_angle(normal, direction);
        REQUIRE(test_result > 0);

        direction = {-6.13546, -6.13546, -10.386499999999998};
        position = {10,10,70};
        ray1 = Ray(direction, position, 1);
        intersection = paraboloid1.get_intersection(ray1);
        normal = paraboloid1.get_normal_vector(intersection);
        test_result = paraboloid1.get_angle(normal, direction);
        REQUIRE(test_result > 0);

        direction = {-6.13546, -6.13546, -10.386499999999998};
        position = {10,10,70};
        ray1 = Ray(direction, position, 1);
        intersection = paraboloid1.get_intersection(ray1);
        normal = paraboloid1.get_normal_vector(intersection);
        test_result = paraboloid1.get_angle(normal, direction);
        REQUIRE(test_result > 0);

        // Inwards of surface
        direction = {3.544393688250574, 2.029649946365913, -10.3864999999999988};
        position = {0.320146311749426, 1.834890053634087, 70};
        ray1 = Ray(direction, position, 1);
        intersection = paraboloid1.get_intersection(ray1);
        normal = paraboloid1.get_normal_vector(intersection);
        test_result = paraboloid1.get_angle(normal, direction);
        REQUIRE(test_result < 0);

        direction = {3.544393688250574, 2.029649946365913, 9.613500000000002};
        position = {0.320146311749426, 1.834890053634087, 50};
        ray1 = Ray(direction, position, 1);
        intersection = paraboloid1.get_intersection(ray1);
        normal = paraboloid1.get_normal_vector(intersection);
        test_result = paraboloid1.get_angle(normal, direction);
        REQUIRE(test_result < 0);

        direction = {1 , 1 ,0};
        position = {0.320146311749426, 1.834890053634087, 50};
        ray1 = Ray(direction, position, 1);
        intersection = paraboloid1.get_intersection(ray1);
        normal = paraboloid1.get_normal_vector(intersection);
        test_result = paraboloid1.get_angle(normal, direction);
        REQUIRE(test_result < 0);
    }

    SECTION("ray_trace method") {
        sixte::SixteVector direction{-0.134093650863128, -3.016847582881728, -6.106675602415699};
        Point_3 position{0.1,0.1,23};
        auto ray1 = Ray(direction, position, 1);
        auto intersection = paraboloid1.get_intersection(ray1);
        auto normal = paraboloid1.get_normal_vector(intersection);
        std::optional<Ray> test_ray = paraboloid1.ray_trace(ray1);
        REQUIRE(test_ray.has_value() == true);

        // area of no real mirror
        direction={-1, -1, 1};
        position={0.1,0.1,23};
        ray1 = Ray(direction, position, 1);
        intersection = paraboloid1.get_intersection(ray1);
        normal = paraboloid1.get_normal_vector(intersection);
        test_ray = paraboloid1.ray_trace(ray1);
        REQUIRE(test_ray.has_value() == false);

    }

}

TEST_CASE("Ray-tracing Hyperboloid", "[ray_tracing_hyperboloid]") {
    Hyperboloid_parameters hyperboloidParameters{1, 1, 1, 20, 10, 1};
    auto hyperboloid = Hyperboloid(hyperboloidParameters);
    SECTION("Intersection Easy") {
        sixte::SixteVector direction{0, 0, -1};
        Point_3 position{0, 0, 4};
        auto ray1 = Ray(direction, position, 1);
        Point_3 intersection_result{0, 0, 2};
        Point_3 test_result = hyperboloid.get_intersection(ray1);
        REQUIRE_THAT(CGAL::to_double(intersection_result.x()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.x()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result.y()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.y()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result.z()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.z()), 1e-9));
    }
    SECTION("Intersection from direction vector x, y != 0") {
        sixte::SixteVector direction{    -6.212387268176032, -2.023246886410702, -4.831784083634364};
        Point_3 position{3.819982828942177, -1.541689096806033, 10.24};
        auto ray1 = Ray(direction, position, 1);
        Point_3 intersection_result{-2.392404439233855, -3.564935983216734, 5.408215916365636};
        Point_3 test_result = hyperboloid.get_intersection(ray1);
        REQUIRE_THAT(CGAL::to_double(intersection_result.x()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.x()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result.y()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.y()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result.z()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.z()), 1e-9));

        sixte::SixteVector direction_1{6.282740338923444, 0.628354849604964, -9995.808635070074};
        Point_3 position_1{-3.819982828942177,1.541689096806033,10000.24};
        ray1 = Ray(direction_1, position_1, 1);
        Point_3 intersection_result_1{2.462757509981267, 2.170043946410997, 4.4313649299257207};
        test_result = hyperboloid.get_intersection(ray1);
        REQUIRE_THAT(CGAL::to_double(intersection_result_1.x()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.x()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result_1.y()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.y()), 1e-9));
        REQUIRE_THAT(CGAL::to_double(intersection_result_1.z()),
                     Catch::Matchers::WithinAbs(CGAL::to_double(test_result.z()), 1e-8));
    }

    SECTION("Angle verification") {
        sixte::SixteVector direction{0, 0, -1};
        Point_3 position{-1.022391762850082, -1.137856058472948, 3.827567051176862};
        auto ray1 = Ray(direction, position, 1);
        double angle_result = 50.0700657603;
        auto intersection = hyperboloid.get_intersection(ray1);
        auto normal = hyperboloid.get_normal_vector(intersection);
        double test_result = hyperboloid.get_angle(normal, direction);

        REQUIRE_THAT(angle_result,  Catch::Matchers::WithinAbs(test_result, 1e-9));

        direction = {0, 0, -1};
        position = {0, 0, 10};
        ray1 = Ray(direction, position, 1);
        angle_result = 90.0;
        intersection = hyperboloid.get_intersection(ray1);
        normal = hyperboloid.get_normal_vector(intersection);
        test_result = hyperboloid.get_angle(normal, direction);
        REQUIRE_THAT(angle_result,  Catch::Matchers::WithinAbs(test_result, 1e-9));

        direction = {0, 0, -1};
        position = {1.022391762850082, 1.137856058472948, 3};
        ray1 = Ray(direction, position, 1);
        angle_result = 50.0700657603;
        intersection = hyperboloid.get_intersection(ray1);
        normal = hyperboloid.get_normal_vector(intersection);
        test_result = hyperboloid.get_angle(normal, direction);
        REQUIRE_THAT(angle_result, Catch::Matchers::WithinAbs(test_result, 1e-9));

        // outwards
        direction = {-0.236186876439912, -1.411128176893316, 1.82757};
        position = {1.258576876439912,2.548988176893316,1};
        ray1 = Ray(direction, position, 1);
        intersection = hyperboloid.get_intersection(ray1);
        normal = hyperboloid.get_normal_vector(intersection);
        test_result = hyperboloid.get_angle(normal, direction);
        REQUIRE(test_result < 0);

        direction = {-10.478867341246726, 97.70732092189037, 49.28702950578477};
        position = {-48.91810227842325, -176.9032804147837, 50.71297049421523};
        ray1 = Ray(direction, position, 1);
        intersection = hyperboloid.get_intersection(ray1);
        normal = hyperboloid.get_normal_vector(intersection);
        test_result = hyperboloid.get_angle(normal, direction);
        REQUIRE(test_result < 0);


        direction = {-10.478867341246726, 97.70732092189037, -57.49580175613352};
        position = {-48.91810227842325, -176.9032804147837, 157.49580175613352};
        ray1 = Ray(direction, position, 1);
        intersection = hyperboloid.get_intersection(ray1);
        normal = hyperboloid.get_normal_vector(intersection);
        test_result = hyperboloid.get_angle(normal, direction);
        REQUIRE(test_result < 0);

        direction = {1, 1, 0};
        position = {-48.91810227842325, -176.9032804147837, 157.49580175613352};
        ray1 = Ray(direction, position, 1);
        intersection = hyperboloid.get_intersection(ray1);
        normal = hyperboloid.get_normal_vector(intersection);
        test_result = hyperboloid.get_angle(normal, direction);
        REQUIRE(test_result < 0);

        // Inwards
        direction = {-59.7171159314194, -81.0308495465274, 49.28702950578477};
        position = {0.320146311749426,1.834890053634087,50.71297049421523};
        ray1 = Ray(direction, position, 1);
        intersection = hyperboloid.get_intersection(ray1);
        normal = hyperboloid.get_normal_vector(intersection);
        test_result = hyperboloid.get_angle(normal, direction);
        REQUIRE(test_result > 0);

        direction = {-49.71984472077189, -73.63531583614764, 43.84050597631358};
        position = {-9.677124898898086, -5.560643656745676, 56.15949402368642};
        ray1 = Ray(direction, position, 1);
        intersection = hyperboloid.get_intersection(ray1);
        normal = hyperboloid.get_normal_vector(intersection);
        test_result = hyperboloid.get_angle(normal, direction);
        REQUIRE(test_result > 0);

        direction = {1,1,0};
        position = {-9.677124898898086, -5.560643656745676, 56.15949402368642};
        ray1 = Ray(direction, position, 1);
        intersection = hyperboloid.get_intersection(ray1);
        normal = hyperboloid.get_normal_vector(intersection);
        test_result = hyperboloid.get_angle(normal, direction);
        REQUIRE(test_result > 0);
    }

    SECTION("ray_trace method") {
        // Out of real mirror range (10 to 20 in z axis as defined up in hyperboloid)
        sixte::SixteVector direction = {1 , 1 ,0};
        Point_3 position = {-5,-5,22.942374290900732};
        Ray ray1 = Ray(direction, position, 1);
        std::optional<Ray> test_ray = hyperboloid.ray_trace(ray1);
        REQUIRE(test_ray.has_value() == false);

        direction = {3.531434678329657, 1.533837904799439, -18.04738215044424};
        position = {-5,-5,22.942374290900732};
        ray1 = Ray(direction, position, 1);
        test_ray = hyperboloid.ray_trace(ray1);
        REQUIRE(test_ray.has_value() == false);

        // In real area of mirror,
        direction = {6.663982824522448, -6.430986631243071, -10.347707663353244};
        position = {-5,-5,22.942374290900732};
        ray1 = Ray(direction, position, 1);
        test_ray = hyperboloid.ray_trace(ray1);
        REQUIRE(test_ray.has_value() == true);

    }

    SECTION("Ray_trace") {

        Paraboloid_parameters paraboloidParameters{4.8698250786416741, 1.6058531031684582, 176.33962264150944, 3184.2948474700024, 3334.2948474700024, 180.27362683400708};
        Paraboloid paraboloid{paraboloidParameters};
        Point_3 position{42.008226, 168.825323, 15000.000000};
        sixte::SixteVector direction{0.000400, 0.000400, -1.000000};
        Ray ray{direction, position, 1};
        std::optional<Ray> result_ray = paraboloid.ray_trace(ray);
        if (result_ray.has_value())
            std::cout <<  std::setprecision(20) <<result_ray.value().position << result_ray.value().direction.x() << ", " << result_ray.value().direction.y() << ", " << result_ray.value().direction.z() << ", " << std::endl;

        hyperboloidParameters={797.56137063995686, 62.416825158873529, 800, 3184.2948474700024, 3034.2948474700024, 0};
        Hyperboloid hyperboloid1{hyperboloidParameters};
        result_ray = hyperboloid1.ray_trace(result_ray.value());
        if (result_ray.has_value())
            std::cout << result_ray.value().position << " " << result_ray.value().direction.x() << ", " << result_ray.value().direction.y() << ", " << result_ray.value().direction.z() << ", " << std::endl;


        Plane sensor = Plane(0, 0, 1, -hyperboloidParameters.c * 2 + (-0.4), 28.8, 28.8);
    }
}*/