/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "NewPSF.h"
#include "test_photon.h"

using namespace sixte;

/* Test parameters:
 * PSF:
    energies [keV]: 0.2, 0.35, 1, 2.5, 7, 10
    thetas [arcmin]: 0, 2, 6, 12, 20, 28
    phis [arcmic]: 0
 * Test parameters:
    Energy [keV]: 2.5, Sky position (RA, Dec) [rad]: (0, 0),
      --> Detector position (x, y, z) [m]: (0.000350011, -0.000237188, 0)
    Energy [keV]: 7.5, Sky position (RA, Dec) [rad]: (0, 0),
      --> Detector position (x, y, z) [m]: (-0.000140766, -1.03419e-05, 0)
    Energy [keV]: 7.5, Sky position (RA, Dec) [rad]: (0.00204, 0.00204),
      --> Detector position (x, y, z) [m]: (0.0248571, -0.0240884, 0)
*/

TEST_CASE("PSF Test", "[psf]") {
  setenv("SIXTE_USE_PSEUDO_RNG","1",1);
  sixte::initSixteRng(1);
  
  std::string filename = "data/instruments/dummy/dummy_psf_voigt_athena_20231120.fits";
  
  double focal_length = 12.;
  
  std::string vig_filename = "data/instruments/dummy/dummy_vig_athena_20231120.fits";
  NewPSF test_psf(filename, focal_length, vig_filename);
  
  std::vector<double> test_energy{0.2, 0.35, 1, 2.5, 7, 10};
  std::vector<double> test_theta{0, 2, 6, 12, 20, 28};
  for (auto& theta : test_theta) theta *= M_PI / 180. / 60.;
  std::vector<double> test_phi{0};
  for (auto& phi : test_phi) phi *= M_PI / 180.;
  
  for (size_t ii=0; ii<test_energy.size(); ii++) {
    REQUIRE_THAT(test_psf.psf_energy()[ii], Catch::Matchers::WithinAbs(test_energy[ii], 1e-9));
  }
  for (size_t ii=0; ii<test_theta.size(); ii++) {
    REQUIRE_THAT(test_psf.psf_theta()[ii], Catch::Matchers::WithinAbs(test_theta[ii], 1e-9));
  }
  for (size_t ii=0; ii<test_phi.size(); ii++) {
    REQUIRE_THAT(test_psf.psf_phi()[ii], Catch::Matchers::WithinAbs(test_phi[ii], 1e-9));
  }
  
  SECTION("get_NewPSF_position") {
    SixtePhoton src_photon_1(0, 1.29, std::make_pair(0, 0), PhotonMetainfo(1, 1));
    SixtePhoton src_photon_2(0, 5.56, std::make_pair(0, 0), PhotonMetainfo(2, 1));
    SixtePhoton src_photon_3(0, 7.91, std::make_pair(0.0333*M_PI/180., 0.0333*M_PI/180.), PhotonMetainfo(3, 1));
    std::vector<SixtePhoton> src_photon_vec{src_photon_1, src_photon_2, src_photon_3};

    Telescope_attitude test_telescope_attitude_1(SixteVector(-0.000111840623, -0.000000054498,  0.999999993746),
                                                 SixteVector( 0.000487279157, -0.999999881280,  0.000000000000),
                                                 SixteVector( 0.999999875025,  0.000487279154,  0.000111840637));
    Telescope_attitude test_telescope_attitude_2(SixteVector(-0.000012691486, -0.000000005567,  0.999999999919),
                                                 SixteVector( 0.000438614432, -0.999999903809,  0.000000000000),
                                                 SixteVector( 0.999999903728,  0.000438614432,  0.000012691487));
    Telescope_attitude test_telescope_attitude_3(SixteVector( 0.000307576497, -0.000000072410,  0.999999952698),
                                                 SixteVector(-0.000235420252, -0.999999972289,  0.000000000000),
                                                 SixteVector( 0.999999924987, -0.000235420241, -0.000307576505));
    std::vector<Telescope_attitude> tel_att_vec{test_telescope_attitude_1, test_telescope_attitude_2, test_telescope_attitude_3};

    std::optional<Point_2> test_pos1 = std::make_optional<Point_2>(-0.002726292905, 0.005786167534);
    std::optional<Point_2> test_pos2 = std::make_optional<Point_2>(-0.000137888942, 0.005122969625);
    std::optional<Point_2> test_pos3 = std::make_optional<Point_2>(0.010890100037, -0.009539417094);
    std::vector<std::optional<Point_2>> test_positions{test_pos1, test_pos2, test_pos3};

    for (size_t ii=0; ii<src_photon_vec.size(); ii++) {
      std::optional<Point_2> get_NewPSF_position_test = test_psf.get_NewPSF_pos(src_photon_vec[ii], tel_att_vec[ii], focal_length);
      std::optional<Point_2> test_pos = test_positions[ii];
      
      REQUIRE_THAT(get_NewPSF_position_test->x(), Catch::Matchers::WithinAbs(test_pos->x(), 1e-9));
      REQUIRE_THAT(get_NewPSF_position_test->y(),
                   Catch::Matchers::WithinAbs(test_pos->y(), 1e-9));
    }
  }
  sixt_destroy_rng();
}
