/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "Pixel.h"

using namespace sixte;

TEST_CASE("Pixel Test", "[pixel]") {
  /** test_requirements */

  std::string xml_filename = "data/instruments/athena-wfi/ld_wfi_ff_large.xml";
  XMLData xml_data(xml_filename);
  std::string xml_filename2 = "data/instruments/athena-xifu/xifu_nofilt_infoc_20240326.xml";
  XMLData xml_data_2(xml_filename2);

  double frame_duration = 9.8e-6;
  T_PixId pix_id = 76600;

  auto num_lines = xml_data.child("detector").child("dimensions").attributeAsInt("ywidth");
  std::vector<unsigned int> lines_of_pix_ids;
  lines_of_pix_ids.reserve(num_lines);
  for (int line = 0; line < num_lines; line++) lines_of_pix_ids.push_back(line);
  std::shared_ptr<LineInfo> line_info(std::make_shared<LineInfo>(num_lines, lines_of_pix_ids));
  line_info->setLastReadoutTime(0, 4);

  auto ph_id = 4;
  auto src_id = 1;
  PhotonMetainfo photon_metainfo(ph_id, src_id);

  //signal pixel should have
  auto signal_val = 93.4;
  auto creation_time = 0.003;
  Signal signal(signal_val, photon_metainfo, creation_time);


    /** test constructor */
  SimplePixel simple_pixel;
  DEPFETPixel depfet_pixel(xml_data, frame_duration, pix_id, line_info);
  EvtPixel evt_pixel(xml_data_2);


  SECTION("Add Signal") {
    /** test values: */

    /** function to be tested: */
    simple_pixel.addSignal(signal);
//    depfet_pixel.addSignal(signal);
    evt_pixel.addSignal(signal);

    /** test: */
    REQUIRE_THAT(simple_pixel.getSignal()->val(), Catch::Matchers::WithinAbs(signal_val, 1e-9));
//    REQUIRE_THAT(depfet_pixel.getSignal()->val(), Catch::Matchers::WithinAbs(signal_val, 1e-9));
    REQUIRE_THAT(evt_pixel.getSignal()->val(), Catch::Matchers::WithinAbs(signal_val, 1e-9));

    REQUIRE(simple_pixel.getSignal()->photon_metainfo()[0].ph_id_ == ph_id);
//    REQUIRE(depfet_pixel.getSignal()->photon_metainfo()[0].ph_id_ == ph_id);
    REQUIRE(evt_pixel.getSignal()->photon_metainfo()[0].ph_id_ == ph_id);
  }

  SECTION("Set Signal") {
    /** test values: */

    /** function to be tested: */
    simple_pixel.setSignal(signal);
//    depfet_pixel.setSignal(signal);
    evt_pixel.setSignal(signal);

    /** test: */
    REQUIRE_THAT(simple_pixel.getSignal()->val(), Catch::Matchers::WithinAbs(signal_val, 1e-9));
//    REQUIRE_THAT(depfet_pixel.getSignal()->val(), Catch::Matchers::WithinAbs(signal_val, 1e-9));
    REQUIRE_THAT(evt_pixel.getSignal()->val(), Catch::Matchers::WithinAbs(signal_val, 1e-9));
  }

  SECTION("Any Signal") {
    /** test values: */
    simple_pixel.addSignal(signal);
//    depfet_pixel.addSignal(signal);
    evt_pixel.addSignal(signal);

    /** function to be tested: */
    bool has_signal_1 = simple_pixel.anySignal();
//    bool has_signal_2 = depfet_pixel.anySignal();
    bool has_signal_3 = evt_pixel.anySignal();

    /** test: */
    REQUIRE(has_signal_1 == true);
//    REQUIRE(has_signal_2 == true);
    REQUIRE(has_signal_3 == true);
  }

  SECTION("Get Signal") {
    /** test values: */
    simple_pixel.addSignal(signal);
//    depfet_pixel.addSignal(signal);
    evt_pixel.addSignal(signal);

    /** function to be tested: */
    const auto& signal_1 = simple_pixel.getSignal();
//    auto signal_2 = depfet_pixel.getSignal();
    const auto& signal_3 = evt_pixel.getSignal();

    /** test: */
    REQUIRE_THAT(signal_1->val(), Catch::Matchers::WithinAbs(signal_val, 1e-9));
//    REQUIRE_THAT(signal_2->val(), Catch::Matchers::WithinAbs(signal_val, 1e-9));
    REQUIRE_THAT(signal_3->val(), Catch::Matchers::WithinAbs(signal_val, 1e-9));
  }

  SECTION("Scale Signal") {
    /** test values: */
    double scaling_factor = 4.4;
    double new_signal = signal_val * scaling_factor;
    simple_pixel.addSignal(signal);
//    depfet_pixel.addSignal(signal);
    evt_pixel.addSignal(signal);

    /** function to be tested: */
    simple_pixel.scaleSignal(scaling_factor);
//    depfet_pixel.scaleSignal(scaling_factor);
    evt_pixel.scaleSignal(scaling_factor);

    /** test: */
    REQUIRE_THAT(simple_pixel.getSignal()->val(), Catch::Matchers::WithinAbs(new_signal, 1e-9));
//    REQUIRE_THAT(depfet_pixel.getSignal()->val(), Catch::Matchers::WithinAbs(new_signal, 1e-9));
    REQUIRE_THAT(evt_pixel.getSignal()->val(), Catch::Matchers::WithinAbs(new_signal, 1e-9));
  }

  SECTION("Clear Signal") {
    /** test values: */
    simple_pixel.addSignal(signal);
//    depfet_pixel.addSignal(signal);
    evt_pixel.addSignal(signal);

    REQUIRE(simple_pixel.anySignal() == true);
//    REQUIRE(depfet_pixel.anySignal() == true);
    REQUIRE(evt_pixel.anySignal() == true);

    /** function to be tested: */
    simple_pixel.clearSignal();
//    depfet_pixel.clearSignal();
    evt_pixel.clearSignal();

    /** test: */
    REQUIRE(simple_pixel.anySignal() == false);
//    REQUIRE(depfet_pixel.anySignal() == false);
    REQUIRE(evt_pixel.anySignal() == false);
  }

}