/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "PhotonProjection.h"
#include "sixte_random.h"

using namespace sixte;

TEST_CASE("Photon Projection Test", "[photon_projection]") {
  sixte::initSixteRng(-1);
  
  std::string xml_filename = "data/instruments/dummy/default_extended.xml";
  pugi::xml_document xml_file;
  xml_file.load_file(xml_filename.c_str());
  XMLData xml_data(xml_file, "data/instruments/dummy/");
  
  PhotonProjection photon_projection (xml_data);
  
  SECTION("Project Event") {
    try {
      size_t rawx = 32;
      size_t rawy = 413;
      
      SixteVector nx(23.3, 4, 8743.4);
      SixteVector ny(4328, 12, 321.4);
      SixteVector nz(12.3, 90.3, 2.4);
      Telescope_attitude telescope_attitude(nx, ny, nz);
      
      double test_reference_ra_lower = 1.35653 - 1e-6;
      double test_reference_ra_upper = 1.35703 + 1e-6;
      double test_reference_dec_upper = -0.194144 + 1e-6;
      double test_reference_dec_lower = -0.195187 - 1e-6;
      
      auto ra_dec = photon_projection.projectRectEvent(rawx, rawy, telescope_attitude);
      
      REQUIRE(ra_dec.first < test_reference_ra_upper);
      REQUIRE(ra_dec.first > test_reference_ra_lower);
      REQUIRE(ra_dec.second < test_reference_dec_upper);
      REQUIRE(ra_dec.second > test_reference_dec_lower);
    } catch (const std::exception& e) {
      sixte::printError(e.what());
      sixt_destroy_rng();
    } catch (const CCfits::FitsException& e) {
      sixte::printError(e.message());
    }
  }
  
//  SECTION("TEST") {
//    double xrpix_=256.5;
//    double yrpix_=256.5;
//    double xrval_=0.;
//    double yrval_=0.;
//    double xdelt_=130.e-6;
//    double ydelt_=130.e-6;
//    double focal_length_ = 12.;
//    auto rotation_angle_ = 0 * M_PI / 180.;
//
//    auto cosrota_ = cos(rotation_angle_);
//    auto sinrota_ = sin(rotation_angle_);
//
//    size_t rawx = 32;
//    size_t rawy = 413;
//
//    SixteVector nx(23.3, 4, 8743.4);
//    SixteVector ny(4328, 12, 321.4);
//    SixteVector nz(12.3, 90.3, 2.4);
//    Telescope_attitude telescope_attitude(nx, ny, nz);
//
//    double x_off = 1; //Test for 0: 1.35703;   Test for 1: 1.35653
//    double y_off = 1; //Test for 0: -0.195187; Test for 1: -0.194144
//
//    double detx = ((double)rawx - xrpix_ + 0.5 + x_off) * xdelt_;
//    double dety = ((double)rawy - yrpix_ + 0.5 + y_off) * ydelt_;
//
//    double x_rota = cosrota_ * detx - sinrota_ * dety;
//    double y_rota = sinrota_ * detx + cosrota_ * dety;
//
//    double detpos_x = x_rota + xrval_; // in [m]
//    double detpos_y = y_rota + yrval_; // in [m]
//
//    SixteVector srcpos = telescope_attitude.nz
//      + (detpos_x / focal_length_ * telescope_attitude.nx)
//      + (detpos_y / focal_length_ * telescope_attitude.ny);
//    srcpos = normalizeVector(srcpos);
//
//    auto ra_dec = calculateRaDec(srcpos);
//    SIXTE_OUT(ra_dec.first)
//    SIXTE_OUT(ra_dec.second)
//  }
}
