/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "PhotonGeneration.h"
#include "sixte_random.h"

using namespace sixte;

TEST_CASE("Photon Generation Test", "[photon_generation]") {
  initSixteRng(4);
  /** test_requirements */
  std::vector<std::string> xml_filenames = {"./data/instruments/athena-wfi/ld_wfi_ff_large_all_chips.xml"};
  std::vector<XMLData> xml_datas;
  
  std::string photon_list = "pholist.fits";
  std::vector<std::string> photon_lists;
  std::string string_prefix;
  
  std::vector<ObsInfo> obs_infos;
  
  NewAttitude attitude(1, 55000, 0, 50, 0.16, 0.16, 0.);
  
  ObsPointing pointing_in("", 0.16, 0.16, 0.);
  ObsTime gti_time(55000., 0., 50.);
  GTICollection gti_collection("", gti_time);
  
  for (const auto &xml_file_name : xml_filenames) {
    pugi::xml_document xml_file;
    xml_file.load_file(xml_file_name.c_str());
    XMLData xml_data(xml_file, "./data/instruments/athena-wfi/");
    xml_datas.push_back(xml_data);
    
    std::string tel_name = xml_data.child("telescope").attributeAsStringOr("number", "");
    if (!tel_name.empty()) {
      string_prefix += "tel" + tel_name + "_";
    }
    photon_lists.push_back(string_prefix + photon_list);
    
    ObsInfo obs_info(xml_data, pointing_in, gti_collection);
    obs_infos.push_back(obs_info);
  }
  
  bool clobber = true;
  
  std::string simput_file_list = "./data/simputs/mcrab.fits";
  std::vector<std::string> simput_files = separate_fits_string(simput_file_list, ',');
  
  std::pair<double, double> dt_sim{0, 40.4};
  
  /** test constructor */
  PhotonGeneration photon_generation(photon_lists, clobber, simput_files, xml_datas, obs_infos);
  
  SECTION("Get Next Photon") {
    /** test values: */
    //SixtePhoton test_sixte_photon; // TODO: needs value

    /** function to be tested: */
    std::optional<SixtePhoton> sixte_photon = photon_generation.getNextPhoton(attitude, dt_sim);
    
    if (!sixte_photon) throw SixteException("jfewop");
//      SIXTE_OUT(sixte_photon->time())
//      SIXTE_OUT(sixte_photon->energy())
//      SIXTE_OUT(sixte_photon->photon_metainfo().tel_id_)
//      SIXTE_OUT(sixte_photon->photon_metainfo().ph_id_)
//      SIXTE_OUT(sixte_photon->photon_metainfo().src_id_)
//      SIXTE_OUT(sixte_photon->detector_position()->x())
//      SIXTE_OUT(sixte_photon->detector_position()->y())

/*
* sixte_photon->time() = 0.000299176
sixte_photon->energy() = 2.47547
sixte_photon->photon_metainfo().tel_id_ = 0
sixte_photon->photon_metainfo().ph_id_ = 1
sixte_photon->photon_metainfo().src_id_ = 1
*/
    /** test: */
    //REQUIRE_THAT(sixte_photon->time(), Catch::Matchers::WithinAbs(test_sixte_photon.time(), 1e-09));
    //REQUIRE_THAT(sixte_photon->energy(), Catch::Matchers::WithinAbs(test_sixte_photon.energy(), 1e-09));
  }

  SECTION("Get ARF Filenames") {
    /** test values: */
    std::vector<std::string> test_arf_filenames = {"./data/instruments/athena-wfi/athena_sixte_wfi_wo_filter_v20190122.arf"};

    /** function to be tested: */
    std::vector<std::string> arf_filenames = getARFFilenames(xml_datas);

    /** test: */
    REQUIRE(arf_filenames == test_arf_filenames);
  }

  SECTION("Get ID Of XML with Largest FOV") {
    /** test values: */
    size_t test_largest_fov = 0; // TODO: needs value

    /** function to be tested: */
    size_t largest_fov = getIDofXMLwithLargestFOV(xml_datas);

    /** test: */
    REQUIRE(largest_fov == test_largest_fov);
  }

  SECTION("clean-up") {
    if (std::remove(photon_list.c_str()) != 0) {
      throw SixteException("Error removing file");
    }
  }
}
