/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "NewSIXT.h"

using namespace sixte;

TEST_CASE("New Sixt Test", "[new sixt]") {
  
  SECTION("Get Date and Time") {
    double mjdref = 55000;
    double t = 7389.23;
    std::pair<std::string, std::string> test_date_time_string = sixtGetDateTime(mjdref, t);
    std::pair<std::string, std::string> test_result_date_time_string = std::make_pair("2009-06-18","02:03:09");
    
    REQUIRE(test_date_time_string.first == test_result_date_time_string.first);
    REQUIRE(test_date_time_string.second == test_result_date_time_string.second);
  }
  
  SECTION("Check for Photon being in FOV") {
    SixteVector photon_direction_1(4.7, 7.4, 7.474);
    SixteVector photon_direction_2(0.0, 0.0, 0.1);
    
    SixteVector telescope_nz(8.7, 11.4, 11.474);
    
    double min_align = 12;
    bool fov = isInFov(photon_direction_1, telescope_nz, min_align);
    bool test_fov = true;
    REQUIRE(fov == test_fov);
    
    min_align = 12;
    fov = isInFov(photon_direction_2, telescope_nz, min_align);
    test_fov = false;
    REQUIRE(fov == test_fov);
  }
  
  SECTION("String Operations") {
    std::string test_string = "432,783,96178,23798,362,59,1";
    
    std::vector<std::string> test_string_string = separate_string_to_strings(test_string, ',');
    std::vector<size_t> test_string_size_t = separate_string_to_size_t (test_string, ',');
    std::vector<double> test_string_double = separate_string_to_double (test_string, ',');
    
    std::vector<std::string> test_string_string_result = {"432", "783", "96178", "23798", "362", "59", "1"};
    std::vector<size_t> test_string_size_t_result = {432, 783, 96178, 23798, 362, 59, 1};
    std::vector<double> test_string_double_result = {432, 783, 96178, 23798, 362, 59, 1};
    
    for (size_t ii=0; ii<test_string_string.size(); ii++) {
      REQUIRE(test_string_string[ii] == test_string_string_result[ii]);
      REQUIRE(test_string_size_t[ii] == test_string_size_t_result[ii]);
      REQUIRE_THAT(test_string_double[ii],  Catch::Matchers::WithinAbs(test_string_double_result[ii], 1e-9));
    }
    
    std::string test_string2 = "432, 783";
    removeSubstring(test_string2, ",");
    REQUIRE(test_string2 == "432 783");
    
    removeSubstring(test_string2, "78");
    REQUIRE(test_string2 == "432 3");
  }

  SECTION("Binary Search") {
    std::vector<double> test_vector = {0.1, 1.2, 2.3, 35.1, 70.9, 100.5};
    std::vector<double> test_values    = {-1, 0.1, 0.2, 1.2, 1.4, 100, 100.5, 120};
    std::vector<int> reference_indices = {-1,   0,   0,   1,   1,   4,    -1,  -1};

    for (size_t ii = 0; ii < test_values.size(); ii++) {
      REQUIRE(findIndexBinarySearch(test_vector, test_values[ii]) == reference_indices[ii]);
    }
  }

}