/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "NewImpactfile.h"
#include "NewSIXT.h"
#include "GTICollection.h"
#include "ObsInfo.h"
#include "SixteCCFits.h"
#include "SixtePhoton.h"
#include "SixteVector.h"
#include "XMLData.h"

using namespace sixte;

TEST_CASE("Impactfile Test", "[impactfile]") {
  std::string filename = "test_impactfile.fits";
  bool clobber = true;
  
  std::string xml_filename = "data/instruments/dummy/default_extended.xml";
  pugi::xml_document xml_file;
  xml_file.load_file(xml_filename.c_str());
  XMLData xml_data(xml_file, "data/instruments/dummy/");
  
  std::string gti_file = "NONE";
  std::string attitude_file = "data/instruments/dummy/attitude_lissajous.fits";
  double mjdref = 55000;
  double tstart = 23.433;
  double exposure = 5000.5;
  double pointing_ra = 45.789;
  double pointing_dec = 4.578;
  double rollangle= 23.709;
  ObsTime obs_time(mjdref, tstart, exposure);
  ObsPointing obs_pointing(attitude_file, pointing_ra, pointing_dec, rollangle);
  
  GTICollection gti_collection(gti_file, obs_time);
  ObsInfo obs_info(xml_data, obs_pointing, gti_collection);
  
  std::vector<double> time{0, 1, 2};
  std::vector<double> energy{2.5, 7.5, 7.5};
  std::vector<double> x{0, 0.00204, 50.1};
  std::vector<double> y{0, 0.00204, 43.52};
  std::vector<long> ph{1, 2, 3};
  std::vector<long> src{1, 1, 1};
  std::vector<size_t> tel{0, 1, 2};
  
  
  SECTION("Add Photons To Impact File") {
    ImpactFile impact_file(filename, clobber, obs_info);
    
    for (size_t ii = 0; ii < time.size(); ii++) {
      SixtePhoton src_photon(time[ii],
                             energy[ii],
                             SixtePoint(x[ii], y[ii], 0),
                             PhotonMetainfo(ph[ii], src[ii], tel[ii]));
      impact_file.addImpact2File(src_photon);
    }
  }
  
  SECTION("Read Photons From Impact File") {
    auto inFile = sixteOpenFITSFileRead(filename, "impact file");
    auto& table = inFile->extension(1);
    long row = table.rows();
    
    std::vector<double> photime;
    std::vector<double> phoenergy;
    std::vector<double> phox;
    std::vector<double> phoy;
    std::vector<long>   phoph;
    std::vector<long>   phosrc;

    table.column("TIME").  read(photime,   1, row);
    table.column("ENERGY").read(phoenergy, 1, row);
    table.column("X").     read(phox,      1, row);
    table.column("Y").     read(phoy  ,    1, row);
    table.column("PH_ID"). read(phoph,     1, row);
    table.column("SRC_ID").read(phosrc,    1, row);

    REQUIRE(photime.size()   == 3);
    REQUIRE(phoenergy.size() == 3);
    REQUIRE(photime.size()   == 3);
    REQUIRE(phox.size()      == 3);
    REQUIRE(phoy.size()      == 3);
    REQUIRE(phosrc.size()    == 3);

    for(size_t ii=0; ii<photime.size(); ii++) {
      REQUIRE_THAT(photime[ii],   Catch::Matchers::WithinAbs(photime[ii],   1e-9));
      REQUIRE_THAT(phoenergy[ii], Catch::Matchers::WithinAbs(phoenergy[ii], 1e-9));
      REQUIRE_THAT(phox[ii],      Catch::Matchers::WithinAbs(phox[ii],      1e-9));
      REQUIRE_THAT(phoy[ii],      Catch::Matchers::WithinAbs(phoy[ii],      1e-9));
      REQUIRE_THAT(phoph[ii],     Catch::Matchers::WithinAbs(phoph[ii],     1e-9));
      REQUIRE_THAT(phosrc[ii],    Catch::Matchers::WithinAbs(phosrc[ii],    1e-9));
    }
    
  }
  
  SECTION("clean-up") {
    if (std::remove(filename.c_str()) != 0) {
      throw SixteException("Error removing file");
    }
  }
}