/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "NewAttitude.h"


/*
 * TEST VALUES:
 *
 * Amplitude=0.035
 * SrcRA=53.13
 * SrcDec=-27.8
 * Exposure=5000
 *
 */

using namespace sixte;

TEST_CASE("Attitude Test", "[attitude]") {
  std::string filename = "data/instruments/dummy/attitude_lissajous.fits";
  double dt = 1;
  double mjdref = 55000;
  double tstart = 0;
  double exposure = 5000;
  double tstop = tstart + exposure;
  double ra = 4.7;
  double dec = 7.4;
  double roll_angle = 4.4;
  
  double amplitude = 0.035;
  double ra0 = 53.13;
  double dec0 = -27.8;
  size_t npoints = 1000;
  
  std::vector<double> times = {0, 100, 4321, 5000};
  
  NewAttitude attitude_1(dt, mjdref, tstart, tstop, ra, dec, roll_angle);
  
  NewAttitude attitude_2(filename, dt);
  
  NewAttitude attitude_3(amplitude, ra0, dec0, tstart, tstop, mjdref, dt, npoints);
  
  SECTION("MJDREF") {
    double test_mjdref1 = attitude_1.mjdref();
    double test_mjdref2 = attitude_2.mjdref();
    double test_mjdref3 = attitude_3.mjdref();
    
    REQUIRE_THAT(test_mjdref1, Catch::Matchers::WithinAbs(mjdref, 1e-9));
    REQUIRE_THAT(test_mjdref2, Catch::Matchers::WithinAbs(mjdref, 1e-9));
    REQUIRE_THAT(test_mjdref3, Catch::Matchers::WithinAbs(mjdref, 1e-9));
  }

  SECTION("dt") {
    double test_dt1 = attitude_1.dt();
    double test_dt2 = attitude_2.dt();
    double test_dt3 = attitude_3.dt();
    
    REQUIRE_THAT(test_dt1, Catch::Matchers::WithinAbs(dt, 1e-9));
    REQUIRE_THAT(test_dt2, Catch::Matchers::WithinAbs(dt, 1e-9));
    REQUIRE_THAT(test_dt3, Catch::Matchers::WithinAbs(dt, 1e-9));
  }

  SECTION("Get Current Velocity") {
    double att_start = 10;
    double att_stop = 20;
    std::pair<double, double> test_velocity = attitude_2.getCurrentVelocity(att_start, att_stop);
    std::pair<double, double> test_result_vel = std::make_pair(4.653930664e-05, 8.792877197e-05);

    REQUIRE_THAT(test_velocity.first, Catch::Matchers::WithinAbs(test_result_vel.first, 1e-6));
    REQUIRE_THAT(test_velocity.second, Catch::Matchers::WithinAbs(test_result_vel.second, 1e-6));
  }

  SECTION("Get Roll Angle") {
    double att_roll_angle1 = attitude_1.getRollAngle(0); // == roll_angle;
    double att_roll_angle2 = attitude_2.getRollAngle(0); // == 0;
    
    REQUIRE_THAT(att_roll_angle1, Catch::Matchers::WithinAbs((roll_angle * M_PI / 180.), 1e-9));
    REQUIRE_THAT(att_roll_angle2, Catch::Matchers::WithinAbs(0, 1e-9));
  }
  
  SECTION("Set WCS Current Pointing") {
    SixteVector test_nz(0.5304352791, 0.708071104, -0.4661262986);
    double crval[2], crval_test_results[2];

    struct wcsprm wcs;
    struct wcsprm test_results_wcs;
    wcs.crval = crval;
    test_results_wcs.crval = crval_test_results;
    
    test_results_wcs.crval[0] = 413.162123026339, test_results_wcs.crval[1] = 207.783138540817;
    
    attitude_2.setWCScurrentPointing(test_nz, &wcs);
    REQUIRE_THAT(wcs.crval[0], Catch::Matchers::WithinAbs(test_results_wcs.crval[0], 1e-9));
    REQUIRE_THAT(wcs.crval[1], Catch::Matchers::WithinAbs(test_results_wcs.crval[1], 1e-9));
    
    attitude_3.setWCScurrentPointing(test_nz, &wcs);
    REQUIRE_THAT(wcs.crval[0], Catch::Matchers::WithinAbs(test_results_wcs.crval[0], 1e-9));
    REQUIRE_THAT(wcs.crval[1], Catch::Matchers::WithinAbs(test_results_wcs.crval[1], 1e-9));
    
  }

  SECTION("Get Telescope Nz") {
    std::vector<SixteVector> telescope_nz_values;
    telescope_nz_values.reserve(times.size());
    for (auto time : times) {
      telescope_nz_values.push_back(SixteVector(attitude_1.getTelescopeNz(time)));
    }
    std::vector<SixteVector> test_results_values{SixteVector(0.9883365559, 0.08125605279, 0.1287955982)};
    
    for (size_t ii=0; ii<times.size(); ii++) {
      REQUIRE_THAT(telescope_nz_values[ii].x(), Catch::Matchers::WithinAbs(test_results_values[0].x(), 1e-6));
      REQUIRE_THAT(telescope_nz_values[ii].y(), Catch::Matchers::WithinAbs(test_results_values[0].y(), 1e-6));
      REQUIRE_THAT(telescope_nz_values[ii].z(), Catch::Matchers::WithinAbs(test_results_values[0].z(), 1e-6));
    }
    
    std::vector<SixteVector> telescope_nz_filename;
    telescope_nz_filename.reserve(times.size());
    for (auto time : times) {
      telescope_nz_filename.push_back(SixteVector(attitude_2.getTelescopeNz(time)));
    }
    std::vector<SixteVector> test_results_filename{SixteVector(0.5304441516, 0.7078930108, -0.4663866286),
                                                   SixteVector(0.5304352791, 0.708071104, -0.4661262986),
                                                   SixteVector(0.5312189524, 0.7074072673, -0.4662417643),
                                                   SixteVector(0.5304441516, 0.7078930108, -0.4663866286)};
    
    for (size_t ii=0; ii<times.size(); ii++) {
      REQUIRE_THAT(telescope_nz_filename[ii].x(), Catch::Matchers::WithinAbs(test_results_filename[ii].x(), 1e-6));
      REQUIRE_THAT(telescope_nz_filename[ii].y(), Catch::Matchers::WithinAbs(test_results_filename[ii].y(), 1e-6));
      REQUIRE_THAT(telescope_nz_filename[ii].z(), Catch::Matchers::WithinAbs(test_results_filename[ii].z(), 1e-6));
    }
    
    std::vector<SixteVector> telescope_nz_lis;
    telescope_nz_lis.reserve(times.size());
    for (auto time : times) {
      telescope_nz_lis.push_back(SixteVector(attitude_3.getTelescopeNz(time)));
    }
    std::vector<SixteVector> test_results_lis{SixteVector(0.5304441516, 0.7078930108, -0.4663866286),
                                              SixteVector(0.5304352791, 0.708071104, -0.4661262986),
                                              SixteVector(0.5312189524, 0.7074072673, -0.4662417643),
                                              SixteVector(0.5304441516, 0.7078930108, -0.4663866286)};
    
    for (size_t ii=0; ii<times.size(); ii++) {
      REQUIRE_THAT(telescope_nz_lis[ii].x(), Catch::Matchers::WithinAbs(test_results_lis[ii].x(), 1e-6));
      REQUIRE_THAT(telescope_nz_lis[ii].y(), Catch::Matchers::WithinAbs(test_results_lis[ii].y(), 1e-6));
      REQUIRE_THAT(telescope_nz_lis[ii].z(), Catch::Matchers::WithinAbs(test_results_lis[ii].z(), 1e-6));
    }
  }

  SECTION("Get Telescope Axes") {
    double time = 100;
    Telescope_attitude telescope_attitude = attitude_1.getTelescopeAxes(time);
    Telescope_attitude test_result_telescope_attitude(SixteVector(-0.1216979496, -0.0869832704, 0.9887484613),
                                                      SixteVector(0.09154485951, -0.9928904091, -0.07608005088),
                                                      SixteVector(0.9883365559, 0.08125605279, 0.1287955982));
    
    REQUIRE_THAT(telescope_attitude.nx.x(), Catch::Matchers::WithinAbs(test_result_telescope_attitude.nx.x(), 1e-8));
    REQUIRE_THAT(telescope_attitude.nx.y(), Catch::Matchers::WithinAbs(test_result_telescope_attitude.nx.y(), 1e-8));
    REQUIRE_THAT(telescope_attitude.nx.z(), Catch::Matchers::WithinAbs(test_result_telescope_attitude.nx.z(), 1e-8));

    REQUIRE_THAT(telescope_attitude.ny.x(), Catch::Matchers::WithinAbs(test_result_telescope_attitude.ny.x(), 1e-8));
    REQUIRE_THAT(telescope_attitude.ny.y(), Catch::Matchers::WithinAbs(test_result_telescope_attitude.ny.y(), 1e-8));
    REQUIRE_THAT(telescope_attitude.ny.z(), Catch::Matchers::WithinAbs(test_result_telescope_attitude.ny.z(), 1e-8));
    
    REQUIRE_THAT(telescope_attitude.nz.x(), Catch::Matchers::WithinAbs(test_result_telescope_attitude.nz.x(), 1e-8));
    REQUIRE_THAT(telescope_attitude.nz.y(), Catch::Matchers::WithinAbs(test_result_telescope_attitude.nz.y(), 1e-8));
    REQUIRE_THAT(telescope_attitude.nz.z(), Catch::Matchers::WithinAbs(test_result_telescope_attitude.nz.z(), 1e-8));
    
    Telescope_attitude telescope_attitude2 = attitude_2.getTelescopeAxes(time);
    Telescope_attitude test_result_telescope_attitude2(SixteVector(0.2794673337, 0.373057282, 0.8847181889),
                                                       SixteVector(0.8003351948, -0.5995528133, 0),
                                                       SixteVector(0.5304352791, 0.708071104, -0.4661262986));
    
    REQUIRE_THAT(telescope_attitude2.nx.x(), Catch::Matchers::WithinAbs(test_result_telescope_attitude2.nx.x(), 1e-7));
    REQUIRE_THAT(telescope_attitude2.nx.y(), Catch::Matchers::WithinAbs(test_result_telescope_attitude2.nx.y(), 1e-7));
    REQUIRE_THAT(telescope_attitude2.nx.z(), Catch::Matchers::WithinAbs(test_result_telescope_attitude2.nx.z(), 1e-7));
    
    REQUIRE_THAT(telescope_attitude2.ny.x(), Catch::Matchers::WithinAbs(test_result_telescope_attitude2.ny.x(), 1e-7));
    REQUIRE_THAT(telescope_attitude2.ny.y(), Catch::Matchers::WithinAbs(test_result_telescope_attitude2.ny.y(), 1e-7));
    REQUIRE_THAT(telescope_attitude2.ny.z(), Catch::Matchers::WithinAbs(test_result_telescope_attitude2.ny.z(), 1e-7));
    
    REQUIRE_THAT(telescope_attitude2.nz.x(), Catch::Matchers::WithinAbs(test_result_telescope_attitude2.nz.x(), 1e-7));
    REQUIRE_THAT(telescope_attitude2.nz.y(), Catch::Matchers::WithinAbs(test_result_telescope_attitude2.nz.y(), 1e-7));
    REQUIRE_THAT(telescope_attitude2.nz.z(), Catch::Matchers::WithinAbs(test_result_telescope_attitude2.nz.z(), 1e-7));
    
    Telescope_attitude telescope_attitude3 = attitude_3.getTelescopeAxes(time);
    Telescope_attitude test_result_telescope_attitude3(SixteVector(0.2794673337, 0.373057282, 0.8847181889),
                                                       SixteVector(0.8003351948, -0.5995528133, 0),
                                                       SixteVector(0.5304352791, 0.708071104, -0.4661262986));
    
    REQUIRE_THAT(telescope_attitude3.nx.x(), Catch::Matchers::WithinAbs(test_result_telescope_attitude3.nx.x(), 1e-7));
    REQUIRE_THAT(telescope_attitude3.nx.y(), Catch::Matchers::WithinAbs(test_result_telescope_attitude3.nx.y(), 1e-7));
    REQUIRE_THAT(telescope_attitude3.nx.z(), Catch::Matchers::WithinAbs(test_result_telescope_attitude3.nx.z(), 1e-7));
    
    REQUIRE_THAT(telescope_attitude3.ny.x(), Catch::Matchers::WithinAbs(test_result_telescope_attitude3.ny.x(), 1e-7));
    REQUIRE_THAT(telescope_attitude3.ny.y(), Catch::Matchers::WithinAbs(test_result_telescope_attitude3.ny.y(), 1e-7));
    REQUIRE_THAT(telescope_attitude3.ny.z(), Catch::Matchers::WithinAbs(test_result_telescope_attitude3.ny.z(), 1e-7));
    
    REQUIRE_THAT(telescope_attitude3.nz.x(), Catch::Matchers::WithinAbs(test_result_telescope_attitude3.nz.x(), 1e-7));
    REQUIRE_THAT(telescope_attitude3.nz.y(), Catch::Matchers::WithinAbs(test_result_telescope_attitude3.nz.y(), 1e-7));
    REQUIRE_THAT(telescope_attitude3.nz.z(), Catch::Matchers::WithinAbs(test_result_telescope_attitude3.nz.z(), 1e-7));
  }
//
//  SECTION("Convert Galactic LB to RA and Dec") {
//    std::vector<double> world;
//    convert_galLB2RAdec(world);
//  }
}