/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "NewArf.h"
#include "sixte_random.h"

/*
 * TEST VALUES:
 *
 * EffArea[0] = 12.6282024384,
 * EffArea[200] = 2153.4692382812;
 * EffArea[500]: = 451.0711059570,
 * EffArea[1023]: = 3.9044334888
 *
 * cumulArf[0][0] = 0.0178754330,
 * cumulArf[200][0] = 0.1385981888,
 * cumulArf[400][4] = 0.7135891914,
 * cumulArf[1023][6] = 1.0000000000
 *
 * test energies for telescope ID search: {1, 3, 6};
 * resulting telescope IDs: {2, 6, 5};
 * (for random numbers: 0.417022, 0.997185, 0.720325; produced with Seed=1)
 */

using namespace sixte;

TEST_CASE("ARF Test", "[arf]") {
  setenv("SIXTE_USE_PSEUDO_RNG","1",1);
  sixte::initSixteRng(1);
  std::vector<std::string> arf_filenames = {"data/instruments/eRosita/tm1_arf_filter_000101v02.fits",
                                            "data/instruments/eRosita/tm2_arf_filter_000101v02.fits",
                                            "data/instruments/eRosita/tm3_arf_filter_000101v02.fits",
                                            "data/instruments/eRosita/tm4_arf_filter_000101v02.fits",
                                            "data/instruments/eRosita/tm5_arf_filter_000101v02.fits",
                                            "data/instruments/eRosita/tm6_arf_filter_000101v02.fits",
                                            "data/instruments/eRosita/tm7_arf_filter_000101v02.fits"};
  NewArf arf(arf_filenames);
  //TODO: test single arf
  
  SECTION("Effective Area") {
    double test_eff_area1 = arf.c_arf()->EffArea[0];
    double test_eff_area2 = arf.c_arf()->EffArea[200];
    double test_eff_area3 = arf.c_arf()->EffArea[500];
    double test_eff_area4 = arf.c_arf()->EffArea[1023];
    
    std::vector<double> test_results{12.6282024384, 2153.4692382812, 451.0711059570, 3.9044334888};
    
    REQUIRE_THAT(test_eff_area1, Catch::Matchers::WithinAbs(test_results[0], 1e-9));
    REQUIRE_THAT(test_eff_area2, Catch::Matchers::WithinAbs(test_results[1], 1e-9));
    REQUIRE_THAT(test_eff_area3, Catch::Matchers::WithinAbs(test_results[2], 1e-9));
    REQUIRE_THAT(test_eff_area4, Catch::Matchers::WithinAbs(test_results[3], 1e-9));
  }
  
  SECTION("Cumulative ARF") {
    double test_cumul_arf_1 = arf.retCumulARF()[0][0];
    double test_cumul_arf_2 = arf.retCumulARF()[200][0];
    double test_cumul_arf_3 = arf.retCumulARF()[400][4];
    double test_cumul_arf_4 = arf.retCumulARF()[1023][6];
    
    std::vector<double> test_results{0.0178754330, 0.1385981888, 0.7135891914, 1.0000000000};
    
    REQUIRE_THAT(test_cumul_arf_1, Catch::Matchers::WithinAbs(test_results[0], 1e-6));
    REQUIRE_THAT(test_cumul_arf_2, Catch::Matchers::WithinAbs(test_results[1], 1e-6));
    REQUIRE_THAT(test_cumul_arf_3, Catch::Matchers::WithinAbs(test_results[2], 1e-6));
    REQUIRE_THAT(test_cumul_arf_4, Catch::Matchers::WithinAbs(test_results[3], 1e-6));
  }

  SECTION("Find ARF Bin Indices") {
    REQUIRE_THROWS(arf.findArfBinIndex(0.01));
    REQUIRE_THROWS(arf.findArfBinIndex(100));

    std::vector<double> test_energies    = {0.1, 0.106, 6.345, 11.99};
    std::vector<double> test_bin_indices = {0  ,     1,   635, 1023};

    for (size_t ii = 0; ii < test_energies.size(); ii++) {
      REQUIRE(arf.findArfBinIndex(test_energies[ii]) == test_bin_indices[ii]);
    }
  }

  SECTION("Get Telescope ID") {
    std::vector<double> test_energy = {1, 3, 6};
    std::vector<double> test_tel_id = {2, 6, 5};
    
    size_t tel_id1 = arf.getTelescopeID(test_energy[0]);
    size_t tel_id2 = arf.getTelescopeID(test_energy[1]);
    size_t tel_id3 = arf.getTelescopeID(test_energy[2]);
    
    REQUIRE_THAT(tel_id1, Catch::Matchers::WithinAbs(test_tel_id[0], 1e-9));
    REQUIRE_THAT(tel_id2, Catch::Matchers::WithinAbs(test_tel_id[1], 1e-9));
    REQUIRE_THAT(tel_id3, Catch::Matchers::WithinAbs(test_tel_id[2], 1e-9));
  }
  
}