/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "Absorber.h"

#include "test_photon.h"
#include "sixte_random.h"

using namespace sixte;

TEST_CASE("Absorber Test", "[absorber]") {
  Absorber absorber;
  initSixteRng(4342);
  
  std::string xml_filename = "data/instruments/dummy/default_extended.xml";
  pugi::xml_document xml_file;
  xml_file.load_file(xml_filename.c_str());
  XMLData xml_data(xml_file, "./data/instruments/dummy/");
  
  Test_Photon test_photon;
  SixtePhoton photon = test_photon.return_sixte_photon("det");
  
  BoundingBox2d bounding_box(Point_2 (-100,-100), Point_2 (100,100));
  
  absorber.setChipId(1);
  absorber.setGeometry(xml_data,bounding_box);
  
  SECTION("set Photon Interaction Strategy") {
    absorber.setPhotonInteractionStrategy(std::make_unique<FullAbsorption>());
    auto rmf = std::make_shared<NewRMF>(xml_data.dirname() + xml_data.child("detector").child("rmf").attributeAsString("filename"));
    absorber.setPhotonInteractionStrategy(std::make_unique<RMFBasedInteraction>(rmf));
  }
  
  SECTION("set Carrier Interaction Strategy") {
    absorber.setCarrierFormationStrategy(std::make_unique<HeatPulseSimple>());
    absorber.setCarrierFormationStrategy(std::make_unique<GaussianSimple>(xml_data));
  }
  
  SECTION("Absorbtion: rmf-based, gaussian") {
    auto rmf = std::make_shared<NewRMF>(xml_data.dirname() + xml_data.child("detector").child("rmf").attributeAsString("filename"));
    absorber.setPhotonInteractionStrategy(std::make_unique<RMFBasedInteraction>(rmf));
    absorber.setCarrierFormationStrategy(std::make_unique<GaussianSimple>(xml_data));
    Carriers test_carriers = absorber.absorb(photon);
    REQUIRE(!test_carriers.empty());
  }
  
  SECTION("Absorbtion: full absorbtion, heat pulse") {
    absorber.setPhotonInteractionStrategy(std::make_unique<FullAbsorption>());
    absorber.setCarrierFormationStrategy(std::make_unique<HeatPulseSimple>());
    Carriers test_carriers = absorber.absorb(photon);
    REQUIRE(!test_carriers.empty());
  }
  
  SECTION("Absorbtion: rmf-based, heat pulse") {
    auto rmf = std::make_shared<NewRMF>(xml_data.dirname() + xml_data.child("detector").child("rmf").attributeAsString("filename"));
    absorber.setPhotonInteractionStrategy(std::make_unique<RMFBasedInteraction>(rmf));
    absorber.setCarrierFormationStrategy(std::make_unique<HeatPulseSimple>());
    Carriers test_carriers = absorber.absorb(photon);
    REQUIRE(!test_carriers.empty());
  }
  
  SECTION("Absorbtion: full absorbtion, gaussian") {
    absorber.setPhotonInteractionStrategy(std::make_unique<FullAbsorption>());
    absorber.setCarrierFormationStrategy(std::make_unique<GaussianSimple>(xml_data));
    Carriers test_carriers = absorber.absorb(photon);
    REQUIRE(!test_carriers.empty());
  }
}
