/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_floating_point.hpp>

#include "Geometry.h"

using namespace sixte;

TEST_CASE("Geometry Test", "[geometry]") {
  SixtePoint pos_cen(0,0,0);
  SixtePoint pos_xoff(1,0,0);
  SixtePoint pos_yoff(0,1,0);

  SECTION("Default Position") {
    Geometry g(0.,0.,0.);

    auto c = g.transformFocalToDet(pos_cen);
    auto x = g.transformFocalToDet(pos_xoff);
    auto y = g.transformFocalToDet(pos_yoff);

    REQUIRE_THAT(c.x(), Catch::Matchers::WithinAbs(pos_cen.x(), 1e-09));
    REQUIRE_THAT(c.y(), Catch::Matchers::WithinAbs(pos_cen.y(), 1e-09));
    REQUIRE_THAT(c.z(), Catch::Matchers::WithinAbs(pos_cen.z(), 1e-09));

    REQUIRE_THAT(x.x(), Catch::Matchers::WithinAbs(pos_xoff.x(), 1e-09));
    REQUIRE_THAT(x.y(), Catch::Matchers::WithinAbs(pos_xoff.y(), 1e-09));
    REQUIRE_THAT(x.z(), Catch::Matchers::WithinAbs(pos_xoff.z(), 1e-09));

    REQUIRE_THAT(y.x(), Catch::Matchers::WithinAbs(pos_yoff.x(), 1e-09));
    REQUIRE_THAT(y.y(), Catch::Matchers::WithinAbs(pos_yoff.y(), 1e-09));
    REQUIRE_THAT(y.z(), Catch::Matchers::WithinAbs(pos_yoff.z(), 1e-09));
  }

  SECTION("Rotated Position") {
    // rotate by 30 deg
    double rota = 30.*M_PI/180.;
    Geometry g(0.,0.,rota);

    auto c = g.transformFocalToDet(pos_cen);
    auto x = g.transformFocalToDet(pos_xoff);
    auto y = g.transformFocalToDet(pos_yoff);

    // point in center is not changed at all
    REQUIRE_THAT(c.x(), Catch::Matchers::WithinAbs(pos_cen.x(), 1e-09));
    REQUIRE_THAT(c.y(), Catch::Matchers::WithinAbs(pos_cen.y(), 1e-09));
    REQUIRE_THAT(c.z(), Catch::Matchers::WithinAbs(pos_cen.z(), 1e-09));

    double xx = cos(rota) * pos_xoff.x() + sin(rota) * pos_xoff.y();
    double yy = -sin(rota) * pos_xoff.x() + cos(rota) * pos_xoff.y();
    REQUIRE_THAT(x.x(), Catch::Matchers::WithinAbs(xx, 1e-09));
    REQUIRE_THAT(x.y(), Catch::Matchers::WithinAbs(yy, 1e-09));
    REQUIRE_THAT(x.z(), Catch::Matchers::WithinAbs(pos_xoff.z(), 1e-09));

    xx = cos(rota) * pos_yoff.x() + sin(rota) * pos_yoff.y();
    yy = -sin(rota) * pos_yoff.x() + cos(rota) * pos_yoff.y();
    REQUIRE_THAT(y.x(), Catch::Matchers::WithinAbs(xx, 1e-09));
    REQUIRE_THAT(y.y(), Catch::Matchers::WithinAbs(yy, 1e-09));
    REQUIRE_THAT(y.z(), Catch::Matchers::WithinAbs(pos_yoff.z(), 1e-09));
  }

  SECTION("Translated Position") {
    double xrval = 0.5;
    double yrval = -0.5;

    Geometry g(xrval, yrval, 0.);

    auto c = g.transformFocalToDet(pos_cen);
    auto x = g.transformFocalToDet(pos_xoff);
    auto y = g.transformFocalToDet(pos_yoff);

    REQUIRE_THAT(c.x(), Catch::Matchers::WithinAbs(pos_cen.x() - xrval, 1e-09));
    REQUIRE_THAT(c.y(), Catch::Matchers::WithinAbs(pos_cen.y() - yrval, 1e-09));
    REQUIRE_THAT(c.z(), Catch::Matchers::WithinAbs(pos_cen.z(), 1e-09));

    REQUIRE_THAT(x.x(), Catch::Matchers::WithinAbs(pos_xoff.x() - xrval, 1e-09));
    REQUIRE_THAT(x.y(), Catch::Matchers::WithinAbs(pos_xoff.y() - yrval, 1e-09));
    REQUIRE_THAT(x.z(), Catch::Matchers::WithinAbs(pos_xoff.z(), 1e-09));

    REQUIRE_THAT(y.x(), Catch::Matchers::WithinAbs(pos_yoff.x() - xrval, 1e-09));
    REQUIRE_THAT(y.y(), Catch::Matchers::WithinAbs(pos_yoff.y() - yrval, 1e-09));
    REQUIRE_THAT(y.z(), Catch::Matchers::WithinAbs(pos_yoff.z(), 1e-09));
  }

  SECTION("Translated and Rotated") {

    double xrval = 0.5;
    double yrval = -0.5;
    double rota = 30.*M_PI/180.;

    Geometry g(xrval, yrval, rota);

    auto c = g.transformFocalToDet(pos_cen);
    auto x = g.transformFocalToDet(pos_xoff);
    auto y = g.transformFocalToDet(pos_yoff);

    double xx = cos(rota) * (pos_cen.x() - xrval) + sin(rota) * (pos_cen.y() - yrval);
    double yy = -sin(rota) * (pos_cen.x() - xrval) + cos(rota) * (pos_cen.y() - yrval);

    REQUIRE_THAT(c.x(), Catch::Matchers::WithinAbs(xx, 1e-09));
    REQUIRE_THAT(c.y(), Catch::Matchers::WithinAbs(yy, 1e-09));
    REQUIRE_THAT(c.z(), Catch::Matchers::WithinAbs(pos_cen.z(), 1e-09));

    xx = cos(rota) * (pos_xoff.x() - xrval) + sin(rota) * (pos_xoff.y() - yrval);
    yy = -sin(rota) * (pos_xoff.x() - xrval) + cos(rota) * (pos_xoff.y() - yrval);
    REQUIRE_THAT(x.x(), Catch::Matchers::WithinAbs(xx, 1e-09));
    REQUIRE_THAT(x.y(), Catch::Matchers::WithinAbs(yy, 1e-09));
    REQUIRE_THAT(x.z(), Catch::Matchers::WithinAbs(pos_xoff.z(), 1e-09));

    xx = cos(rota) * (pos_yoff.x() - xrval) + sin(rota) * (pos_yoff.y() - yrval);
    yy = -sin(rota) * (pos_yoff.x() - xrval) + cos(rota) * (pos_yoff.y() - yrval);
    REQUIRE_THAT(y.x(), Catch::Matchers::WithinAbs(xx, 1e-09));
    REQUIRE_THAT(y.y(), Catch::Matchers::WithinAbs(yy, 1e-09));
    REQUIRE_THAT(y.z(), Catch::Matchers::WithinAbs(pos_yoff.z(), 1e-09));
  }

  SECTION("Inversion") {

    double xrval = 0.5;
    double yrval = -0.5;
    double rota = 30.*M_PI/180.;

    Geometry g(xrval, yrval, rota);

    auto c = g.transformFocalToDet(pos_cen);
    auto x = g.transformFocalToDet(pos_xoff);
    auto y = g.transformFocalToDet(pos_yoff);

    auto cc = g.transformDetToFocal(c);
    auto xx = g.transformDetToFocal(x);
    auto yy = g.transformDetToFocal(y);

    REQUIRE_THAT(cc.x(), Catch::Matchers::WithinAbs(pos_cen.x(), 1e-09));
    REQUIRE_THAT(cc.y(), Catch::Matchers::WithinAbs(pos_cen.y(), 1e-09));
    REQUIRE_THAT(cc.z(), Catch::Matchers::WithinAbs(pos_cen.z(), 1e-09));

    REQUIRE_THAT(xx.x(), Catch::Matchers::WithinAbs(pos_xoff.x(), 1e-09));
    REQUIRE_THAT(xx.y(), Catch::Matchers::WithinAbs(pos_xoff.y(), 1e-09));
    REQUIRE_THAT(xx.z(), Catch::Matchers::WithinAbs(pos_xoff.z(), 1e-09));

    REQUIRE_THAT(yy.x(), Catch::Matchers::WithinAbs(pos_yoff.x(), 1e-09));
    REQUIRE_THAT(yy.y(), Catch::Matchers::WithinAbs(pos_yoff.y(), 1e-09));
    REQUIRE_THAT(yy.z(), Catch::Matchers::WithinAbs(pos_yoff.z(), 1e-09));


  }
}
