#!/bin/bash
set -e

# Compare output of E2E test for Crab with Athena WFI

### Setup
. ./data/scripts/setup/util_e2e.sh

base_sim=crab_sixtesim
base_ref=crab_sixtesim_ref

simdata_dir=./output/
refdata_dir=./data/refdata/e2e_athena-wfi_crab/
###


###### Eventfile ######
printf "########## Comparing Eventfile ########## \n"

# 1. ftdiff (chip EVENT files, header only)
for ii in {0..3}
do
  ftdiff \
    infile1=${simdata_dir}/chip${ii}_evt_${base_sim}.fits \
    infile2=${refdata_dir}/chip${ii}_evt_${base_ref}_header_only.fits \
    cmpdata=no caldsum=no reltol=0.32 tolerance=25 \
    exclude="PHA2PI,NAXIS2"
done

# 2. ftdiff (merged EVENT file, header only)
ftdiff \
  infile1=${simdata_dir}/evt_${base_sim}.fits \
  infile2=${refdata_dir}/evt_${base_ref}_header_only.fits \
  cmpdata=no caldsum=no reltol=0.32 tolerance=25 \
  exclude="PHA2PI,NEVENTS,NAXIS2"

# 3. Compare total number of events
# Check if abs(num_events_sim-num_events_ref) < 3*sqrt(num_events_ref)
num_events_sim=$(ftkeypar ${simdata_dir}/evt_${base_sim}.fits[1] NAXIS2 chatter=3 | grep 'value:' | awk '{print $2}')
num_events_ref=$(ftkeypar ${refdata_dir}/evt_${base_ref}_header_only.fits[1] NEVENTS chatter=3 | grep 'value:' | awk '{print $2}')

diff=$((num_events_sim - num_events_ref))
abs_diff=${diff#-}  # Get the absolute value
threshold=$(bc <<< "scale=10; 3 * sqrt($num_events_ref)")

printf "\n num_events_sim: $num_events_sim, num_events_ref: $num_events_ref, abs_diff: $abs_diff, threshold: $threshold\n"

if (( $(bc <<< "$abs_diff > $threshold") )); then
    printf "Error: Major difference in number of events\n"
    exit 1
fi


###### Image ######
printf "\n\n\n########## Comparing Image ########## \n"

# 1. ftdiff (header + data)
ftdiff \
  infile1=${simdata_dir}/img_${base_sim}.fits \
  infile2=${refdata_dir}/img_${base_ref}.fits \
  reltol=0.30 tolerance=36 caldsum=no

# 2. fimgstat
# Compare sum of all pixel values (relative difference < 0.5%):
# abs(sum_sim-sum_ref)/sum_ref*100 < 0.5
imgstat_sim=$(fimgstat \
  infile=${simdata_dir}/img_${base_sim}.fits \
  threshlo=INDEF threshup=INDEF)
imgsum_sim=$(echo "$imgstat_sim" | grep "The sum of the selected image" | awk '{print $NF}')

imgstat_ref=$(fimgstat \
  infile=${refdata_dir}/img_${base_ref}.fits \
  threshlo=INDEF threshup=INDEF)
imgsum_ref=$(echo "$imgstat_ref" | grep "The sum of the selected image" | awk '{print $NF}')

# Check if the relative difference is greater than 0.5 percent
imgsum_rel_diff=$(bc <<< "scale=10; ($imgsum_sim - $imgsum_ref) / $imgsum_ref * 100")
imgsum_rel_diff_abs=$(bc <<< "scale=10; sqrt($imgsum_rel_diff^2)")

printf "\nimgsum_sim: $imgsum_sim, imgsum_ref: $imgsum_ref, imgsum_rel_diff_abs: $imgsum_rel_diff_abs\n"

if (( $(bc <<< "$imgsum_rel_diff_abs > 0.5") )); then
  printf "Error: Relative difference of image sum greater than 0.5 percent\n"
  exit 1
fi


###### Spectrum ######
printf "\n\n\n########## Comparing Spectrum ########## \n"

# 1. ftdiff (header + data)
ftdiff \
  infile1=${simdata_dir}/spec_${base_sim}.pha[SPECTRUM] \
  infile2=${refdata_dir}/spec_${base_ref}.pha[SPECTRUM] \
  reltol=0.17 tolerance=40 caldsum=no \
  exclude="ANCRFILE,RESPFILE,PHA2PI"


### Comparison finished successfully
printf "\n%s executed successfully\n" "${0}"
exit 0
