/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>
#include <memory>

namespace sixte {

/**
 * Initializes the global random number generator with rng_seed. If rng_seed
 * is 0, the seed is determined from the system clock in microsecond precision.
 */
void initSixteRng(int rng_seed);

double getGaussRandomNumber();

double getUniformRandomNumber();


// TODO: implement cpp version (and rewrite sixte rng altogether)

class PoissonGSLRng {
 public:
  explicit PoissonGSLRng (int rng_seed) {
    random_number_ = gsl_rng_alloc(gsl_rng_ranlux);
    gsl_rng_set(random_number_, rng_seed);
  }

  ~PoissonGSLRng () {
    gsl_rng_free(random_number_);
  }

  size_t getPoissonGSLRngNumber(double weight) {
    return gsl_ran_poisson(random_number_, weight);
  }

  double getFlatGSLRngNumber(double a, double b) {
    return gsl_ran_flat(random_number_, a, b);
  }

 private:
  gsl_rng* random_number_;
};

extern std::unique_ptr<PoissonGSLRng> poisson_gsl_pointer;

double getFlatGSLRngNumber(double a, double b);
size_t getPoissonGSLRngNumber(double weight);

}

