/*
    This file is part of SIXTE.

    SIXTE is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    SIXTE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    For a copy of the GNU General Public License see
    <http://www.gnu.org/licenses/>.


    Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
    Erlangen-Nuernberg
*/


#include "SurfaceModel.h"

#include "XMLData.h"
#include "raytracing/surface/Dummy.h"
#include "raytracing/surface/GaussSurface.h"
#include "raytracing/surface/Microfacet.h"

bool SurfaceModel::simulate_surface(Ray &ray) const
{
    return surface_strategy_->simulate_surface(ray);
}

void
SurfaceModel::set_surface_parameter(std::string model, std::string shadowing, double factor, double shadowing_factor) {
    return surface_strategy_->set_surface_parameter(model, shadowing, factor, shadowing_factor);
}

std::shared_ptr<SurfaceModel> SurfaceModel::from_xml(const sixte::XMLNode& surface_node) {
    std::string surface_model = surface_node.attributeAsString("model");
    if (surface_model == "gauss") {
        double factor = surface_node.attributeAsDouble("roughness");
        return std::make_shared<SurfaceModel>(std::make_unique<GaussSurface>(factor));
    }
    if (surface_model == "microfacet") {
        double factor = surface_node.attributeAsDouble("roughness");
        double factor_shadowing = surface_node.attributeAsDouble("shadowing_alpha");
        std::string mf_type = surface_node.attributeAsString("type");
        std::string mf_shadowing = surface_node.attributeAsString("shadowing");
        bool ggx = false;
        bool ggx_shadowing = false;
        if (mf_type == "ggx")
            ggx = true;
        if (mf_shadowing == "ggx")
            ggx_shadowing = true;
        return std::make_shared<SurfaceModel>(
            std::make_unique<Microfacet>(factor, factor_shadowing, ggx, ggx_shadowing));
    }
    return std::make_shared<SurfaceModel>(std::make_unique<Dummy>());
}
