/*
This file is part of SIXTE.
   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.
   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.
   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.
   Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "Reflectivity.h"

#include <cmath>
#include <complex>
#include <utility>
#include <vector>
#include "SixteCCFits.h"
#include "SixteVector.h"
#include "sixte_random.h"

namespace sixte {
    SurfaceElement::SurfaceElement() {

    };
    SurfaceElement::SurfaceElement(std::string  database_path,
                                   std::string  element_name)
        : database_path_(std::move(database_path)), element_name_(std::move(element_name)) {
      Fitsfile fitsfile(database_path_, FileMode::read);
      fitsfile.moveToExt(element_name_);

      const auto num_rows = fitsfile.getNumRows();

      fitsfile.readCol("Energy", 1, num_rows, energy_grid_);
      fitsfile.readCol("f1", 1, num_rows, f1_list_);
      fitsfile.readCol("f2", 1, num_rows, f2_list_);
      fitsfile.readKey("Density", density_);
      fitsfile.readKey("Mass", mass_);

      std::vector<std::vector<double>::iterator> grid_iter;
      grid_iter.push_back(energy_grid_.begin());

      std::array<int, 1> grid_size;
      grid_size[0] = energy_grid_.size();
      const int num_elements = grid_size[0];

      interp_ML_f1_ = linterp::InterpMultilinear<1, double>(
          grid_iter.begin(), grid_size.begin(), f1_list_.data(),
          f1_list_.data() + num_elements);
      interp_ML_f2_ = linterp::InterpMultilinear<1, double>(
          grid_iter.begin(), grid_size.begin(), f2_list_.data(),
          f2_list_.data() + num_elements);
    };

    SurfaceElement::SurfaceElement(const SurfaceElement& o)
            : database_path_(o.database_path_),
              element_name_(o.element_name_),
              f1_list_(o.f1_list_),
              f2_list_(o.f2_list_),
              energy_grid_(o.energy_grid_),
              density_(o.density_),
              mass_(o.mass_) {
        std::vector<std::vector<double>::iterator> grid_iter{ energy_grid_.begin() };
        std::array<int,1> grid_size{ static_cast<int>(energy_grid_.size()) };
        interp_ML_f1_.emplace(
                grid_iter.begin(), grid_size.begin(),
                f1_list_.data(), f1_list_.data() + grid_size[0]
        );
        interp_ML_f2_.emplace(
                grid_iter.begin(), grid_size.begin(),
                f2_list_.data(), f2_list_.data() + grid_size[0]
        );
    }

    SurfaceElement& SurfaceElement::operator=(const SurfaceElement& o) {
        if (this != &o) {
            database_path_ = o.database_path_;
            element_name_  = o.element_name_;
            f1_list_       = o.f1_list_;
            f2_list_       = o.f2_list_;
            energy_grid_   = o.energy_grid_;
            density_       = o.density_;
            mass_          = o.mass_;

            std::vector<std::vector<double>::iterator> grid_iter{ energy_grid_.begin() };
            std::array<int,1> grid_size{ static_cast<int>(energy_grid_.size()) };
            interp_ML_f1_.emplace(
                    grid_iter.begin(), grid_size.begin(),
                    f1_list_.data(), f1_list_.data() + grid_size[0]
            );
            interp_ML_f2_.emplace(
                    grid_iter.begin(), grid_size.begin(),
                    f2_list_.data(), f2_list_.data() + grid_size[0]
            );
        }
        return *this;
    }

    double SurfaceElement::reflectivity(const double energy,
                                        const double angle) const {
      using namespace PhysicalConstants;
      if (angle <= 0 || angle >= 90) {
        throw std::out_of_range("Error: Given angle: " + std::to_string(angle) +
                                " is out of range. ");
      }
      std::array<double, 1> arg = {energy};
      const double f1 = interp_ML_f1_->interp(arg.begin());
      const double f2 = interp_ML_f2_->interp(arg.begin());

      const double frac = (N_A * density_ * R_E) / (mass_ * 2 * M_PI);
      const double lamda = (H * C / energy);

      const std::complex<double> n1(1, 0);
      const std::complex<double> n2((1 - frac * (lamda * lamda) * f1),
                                    -frac * (lamda * lamda) * f2);

      const double st = sin(angle);
      const double ct = cos(angle);

      const std::complex<double> sq =
          std::sqrt((1. - (n1 / n2) * st) * (1. + (n1 / n2) * st));
      const std::complex<double> rs = (n1 * ct - n2 * sq) / (n1 * ct + n2 * sq);
      const std::complex<double> rt = (n1 * sq - n2 * ct) / (n1 * sq + n2 * ct);

      return (abs(rs) * abs(rs) + abs(rt) * abs(rt)) / 2.0;
    };

    bool SurfaceElement::doesReflect(const double energy,
                                     const double angle) const {
      const double ref_value_ = reflectivity(energy, angle);
      if (getUniformRandomNumber() < ref_value_) {
        return true;
      } else {
        return false;
      }
    }


}  // namespace sixte
