/*
    This file is part of SIXTE.

    SIXTE is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    SIXTE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    For a copy of the GNU General Public License see
    <http://www.gnu.org/licenses/>.


    Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
    Erlangen-Nuernberg
*/

#include "Wolter.h"
#include <string>

Wolter::Wolter(const sixte::XMLData& xml_data)
    : raytracing_node_(xml_data.child("telescope").child("raytracer")) {
    shapes = EmbreeScene();
  // focal length in mm
  focal_length_ = (xml_data.child("telescope").child("focallength").attributeAsDouble("value") * 1000);
    Wolter::create();
}

std::optional<Ray> Wolter::ray_trace(Ray &ray) {
    return shapes.ray_trace(ray);
}


void Wolter::create_parameters(const double new_radius, Paraboloid_parameters &p_pars, Hyperboloid_parameters &h_pars) const {
    const double local_theta = asin(new_radius/focal_length_)/4;
    p_pars.Xp_min = focal_length_ * cos(4 * local_theta) + 2 * h_pars.c;
    p_pars.Yp_min = new_radius;
    p_pars.p = p_pars.Yp_min * tan(local_theta);
    p_pars.Yp_max = sqrt(p_pars.p *(2*p_pars.Xp_max + p_pars.p));
    h_pars.a = focal_length_ * (2 * cos(2*local_theta) - 1) / 2;
    h_pars.b = sqrt(pow(h_pars.c, 2) - pow(h_pars.a, 2));
}

std::vector<std::string> split(std::string s, const std::string& delimiter) {
    std::vector<std::string> tokens;
    size_t pos = 0;
    std::string token;
    while ((pos = s.find(delimiter)) != std::string::npos) {
        token = s.substr(0, pos);
        tokens.push_back(token);
        s.erase(0, pos + delimiter.length());
    }
    tokens.push_back(s);

    return tokens;
}


void Wolter::create() {
    Paraboloid_parameters p_pars{};
    Hyperboloid_parameters h_pars{};

    std::string telescope_type = raytracing_node_.child("type").attributeAsString("type");
    outer_radius = raytracing_node_.child("type").attributeAsDouble("outer_diameter") / 2;
    inner_radius = raytracing_node_.child("type").attributeAsDouble("inner_diameter") / 2;
    number_of_shells = raytracing_node_.child("type").attributeAsInt("mirror_shells");
    mirror_height = raytracing_node_.child("type").attributeAsDouble("mirror_height");
    sensor_offset = raytracing_node_.child("sensor").attributeAsDouble("offset");
    double sensor_x = raytracing_node_.child("sensor").attributeAsDouble("sensor_x");
    double sensor_y = raytracing_node_.child("sensor").attributeAsDouble("sensor_y");

    std::string spider_flag = raytracing_node_.child("spider").attributeAsString("spider");
    Vec3fa spider_position = {};
    spider_position.x = (float) raytracing_node_.child("spider").attributeAsDouble("position_x");
    spider_position.y = (float) raytracing_node_.child("spider").attributeAsDouble("position_y");
    spider_position.z = (float) raytracing_node_.child("spider").attributeAsDouble("position_z");
    std::string spider_path = raytracing_node_.child("spider").attributeAsString("path");


    if (spider_flag == "true")
        shapes.spider = Spider(spider_path, spider_position);

    std::string material_path = raytracing_node_.child("surface").attributeAsString("material_path");
    std::string material = raytracing_node_.child("surface").attributeAsString("material");

    shapes.mirror_coating = sixte::SurfaceElement(material_path, material);
    auto surface_model = SurfaceModel::from_xml(raytracing_node_.child("surface"));

    const auto mirror = raytracing_node_.child("mirror");
    std::string mirror_flag = mirror.attributeAsString("exact");

    bool first_shell = true;
    double first_shell_height = 0;
    if (mirror_flag == "true") {
        auto shell_positions = split(mirror.attributeAsString("positions"), ",");
        for (const auto& shell : shell_positions) {
            auto Ypmin = std::stod(shell);
            p_pars.theta = asin(Ypmin / focal_length_) / 4;
            h_pars.c = focal_length_ / 2;
            p_pars.Xp_min = focal_length_ * cos(4 * p_pars.theta) + 2 * h_pars.c;
            p_pars.Xp_max = mirror_height + p_pars.Xp_min;
            p_pars.Yp_min = Ypmin;
            p_pars.p = p_pars.Yp_min * tan(p_pars.theta);
            p_pars.Yp_max = sqrt(p_pars.p * (2 * p_pars.Xp_max + p_pars.p));

            h_pars.theta = p_pars.theta;
            h_pars.a = focal_length_ * (2 * cos(2*h_pars.theta) - 1) / 2;
            h_pars.b = sqrt(pow(h_pars.c, 2) - pow(h_pars.a, 2));
            h_pars.Xh_max = p_pars.Xp_min;
            h_pars.Xh_min = p_pars.Xp_min - mirror_height;
            h_pars.Yh_max = p_pars.Yp_min;
            h_pars.Yh_min = h_pars.b * sqrt(pow(h_pars.Xh_min - h_pars.c, 2) / pow(h_pars.a, 2) - 1);

            if (first_shell) {
                /*p_pars.angle_x = 0.03 * M_PI / 180;
                p_pars.angle_y = 0.03 * M_PI / 180;;
                */
                p_pars.origin = Vec3fa(0,0, 0);
                first_shell_height = p_pars.Xp_max;
                first_shell=false;
            } else {
                auto z_offset = (float) (first_shell_height - p_pars.Xp_max);
                p_pars.origin = Vec3fa(0, 0, z_offset);
                h_pars.origin = Vec3fa(0, 0, z_offset);
            }

            p_pars.surface = surface_model;
            h_pars.surface = surface_model;

            shapes.hyperboloids.emplace_back(h_pars);
            shapes.paraboloids.emplace_back(p_pars);

        }

    } else {
        h_pars.c = focal_length_ / 2;
        p_pars.theta = asin(outer_radius / focal_length_) / 4 * 180 / M_PI;
        h_pars.theta = p_pars.theta;

        // Most outer paraboloid
        p_pars.Xp_min = focal_length_ * cos(4 * p_pars.theta) + 2 * h_pars.c;
        p_pars.Xp_max = mirror_height + p_pars.Xp_min;
        p_pars.Yp_min = outer_radius;
        p_pars.p = p_pars.Yp_min * tan(p_pars.theta);
        p_pars.Yp_max = sqrt(p_pars.p * (2 * p_pars.Xp_max + p_pars.p));

        distance_to_mirror = (outer_radius - inner_radius) / (number_of_shells - 1);

        for (int i = 0; i < number_of_shells; i++) {

            create_parameters(outer_radius - distance_to_mirror * i, p_pars, h_pars);
            // p_pars.Xp_min = focal_length_ * cos(4 * p_pars.theta) + 2 * h_pars.c;
            p_pars.Xp_max = mirror_height + p_pars.Xp_min;
            h_pars.Xh_max = p_pars.Xp_min;
            h_pars.Xh_min = p_pars.Xp_min - mirror_height;
            //    p_pars.id = shape_id{'p', (short) i};
            //    h_pars.id = shape_id{'h', (short) i};
            h_pars.Yh_max = p_pars.Yp_min;
            h_pars.Yh_min = h_pars.b * sqrt(pow(h_pars.Xh_min - h_pars.c, 2) / pow(h_pars.a, 2) - 1);

            p_pars.surface = surface_model;
            h_pars.surface = surface_model;

            shapes.hyperboloids.emplace_back(h_pars);
            shapes.paraboloids.emplace_back(p_pars);

        }
    }
    shapes.sensor = Plane{0, 0, 1, -h_pars.c * 2 + sensor_offset, sensor_x, sensor_y};
    shapes.scene = shapes.initializeScene(shapes.device);

}

void Wolter::set_surface_parameter(std::string model, std::string shadowing, double factor, double shadowing_factor) {
    std::cout << "Changing parameter" << std::endl;
    for (auto &paraboloid : shapes.paraboloids) {
        paraboloid.surface->set_surface_parameter(model, shadowing, factor, shadowing_factor);
    }
    for (auto &hyperboloid : shapes.hyperboloids) {
        hyperboloid.surface->set_surface_parameter(model, shadowing, factor, shadowing_factor);
    }

}
