/*
    This file is part of SIXTE.

    SIXTE is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    SIXTE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    For a copy of the GNU General Public License see
    <http://www.gnu.org/licenses/>.


    Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
    Erlangen-Nuernberg
*/

#pragma once


#include <memory>

#include "MirrorModule.h"
#include "raytracing/shape/Spider.h"
#include "raytracing/shape/OpticalMesh.h"
#include "raytracing/shape/Plane.h"
#include "raytracing/shape/Pore.h"
#include <algorithm>
#include <optional>
#include "EmbreeScene.h"

class SurfaceModel;

class LobsterEyeOptic final : public virtual MirrorModule {
 public:
    explicit LobsterEyeOptic(const sixte::XMLData& xml_data);

    ~LobsterEyeOptic() override = default;

    LobsterEyeOptic(const LobsterEyeOptic &o) = default;

    LobsterEyeOptic(LobsterEyeOptic&& o) noexcept = default;

    [[nodiscard]] std::unique_ptr<MirrorModule> clone() const override {
        return std::make_unique<LobsterEyeOptic>(*this);
    }

    std::optional<Ray> ray_trace(Ray& ray) override;
    std::optional<Ray> ray_trace_optics(Ray& ray) override;
    std::optional<Ray> ray_trace_terminal(Ray& ray) override;

    void set_surface_parameter(std::string model, std::string shadowing, double factor, double shadowing_factor) override {};

    [[nodiscard]] double focal_length() const override {
      return focal_length_;
    };

    [[nodiscard]] double entrance_plane_z() const override {
      return 2.0 * focal_length_;
    };

    [[nodiscard]] std::vector<Sensor> mesh_sensor() const {
      return mesh_sensor_;
    };

 private:
  std::string xml_path_;
  sixte::XMLNode raytracer_node_;
  double focal_length_;
  RTCDevice device_;
  RTCScene full_scene_;
  RTCScene optics_scene_;

  Spider spider_;
  OpticalMesh opticalMesh_;
  unsigned int optical_geom_id_full_ = RTC_INVALID_GEOMETRY_ID;
  unsigned int optical_geom_id_optics_ = RTC_INVALID_GEOMETRY_ID;
  Plane sensor_;
  std::vector<Sensor> mesh_sensor_;
  Pore pore_;
  std::shared_ptr<SurfaceModel> surface_model_;

  RTCScene initializeFullScene(RTCDevice device);
  RTCScene initializeOpticsScene(RTCDevice device);

  void create() override;
  void createSpider();
  void createSurface();
  void createOptic();
  void createMPOs();
  void createSensor();
  static Vec3fa readSensorPosition(const sixte::XMLNode& node) ;

  bool embree_ray_trace(Ray &ray, int depth);
  bool embree_ray_trace_optics(Ray &ray, int depth);
  bool embree_ray_trace_terminal(Ray &ray);
  std::optional<size_t> FindSensorIndexByGeomID(size_t geom_id);
};
