/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.

   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "paraminput.h"
#include "ape/pil.h"
#include "SixteException.h"

namespace sixte {

void checkPILStatus(int pil_status, const std::string& parameter_name) {
  if (pil_status != PIL_OK) {
    throw SixteException("Failed to query for parameter \"" + parameter_name +
        "\" (" + PIL_err_handler(pil_status) + ")");
  }
}

// Wrapper function for the Parameter Interface Library API.
template<typename T_Parameter>
T_Parameter PILGetWrapper(const std::string& parameter_name, int& status);

// Template specializations of PILGetWrapper for all supported types.
template<>
int PILGetWrapper<int>(const std::string& parameter_name, int& status) {
  int result;
  status = PILGetInt(parameter_name.c_str(), &result);

  return result;
}

template<>
long PILGetWrapper<long>(const std::string& parameter_name, int& status) {
  long result;
  status = PILGetLong(parameter_name.c_str(), &result);

  return result;
}

template<>
float PILGetWrapper<float>(const std::string& parameter_name, int& status) {
  float result;
  status = PILGetReal4(parameter_name.c_str(), &result);

  return result;
}

template<>
double PILGetWrapper<double>(const std::string& parameter_name, int& status) {
  double result;
  status = PILGetReal(parameter_name.c_str(), &result);

  return result;
}

template<>
std::string PILGetWrapper<std::string>(const std::string& parameter_name, int& status) {
  char tmp_result[PIL_LINESIZE];
  status = PILGetString(parameter_name.c_str(), tmp_result);

  return static_cast<std::string>(tmp_result);
}

template<>
bool PILGetWrapper<bool>(const std::string& parameter_name, int& status) {
  int tmp_result;
  status = PILGetBool(parameter_name.c_str(), &tmp_result);

  return static_cast<bool>(tmp_result);
}

template<typename T_Parameter>
T_Parameter queryParameter(const std::string& parameter_name) {
  int status = PIL_OK;
  T_Parameter result = PILGetWrapper<T_Parameter>(parameter_name, status);
  checkPILStatus(status, parameter_name);

  return result;
}

// User interface functions.
int queryParameterInt(const std::string& parameter_name) {
  return queryParameter<int>(parameter_name);
}

long queryParameterLong(const std::string& parameter_name) {
  return queryParameter<long>(parameter_name);
}

float queryParameterFloat(const std::string& parameter_name) {
  return queryParameter<float>(parameter_name);
}

double queryParameterDouble(const std::string& parameter_name) {
  return queryParameter<double>(parameter_name);
}

std::string queryParameterString(const std::string& parameter_name) {
  return queryParameter<std::string>(parameter_name);
}

bool queryParameterBool(const std::string& parameter_name) {
  return queryParameter<bool>(parameter_name);
}

}