/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "TemplateFitsFiles.h"

namespace sixte {

std::unique_ptr<CCfits::FITS> EventFileTemplate(const std::string& filename) {
  std::unique_ptr<CCfits::FITS> inFile;
  inFile = std::make_unique<CCfits::FITS>(filename, CCfits::Write);
  
  std::vector<std::string> colNames {"TIME",
                                     "FRAME",
                                     "PHA",
                                     "PI",
                                     "SIGNAL",
                                     "RAWX",
                                     "RAWY",
                                     "RA",
                                     "DEC",
                                     "PH_ID",
                                     "SRC_ID",
                                     "TYPE",
                                     "NPIXELS",
                                     "PILEUP",
                                     "SIGNALS",
                                     "PHAS"};
  
  std::vector<std::string> colForms {"D",
                                     "J",
                                     "J",
                                     "J",
                                     "E",
                                     "I",
                                     "I",
                                     "D",
                                     "D",
                                     "2J",
                                     "2J",
                                     "I",
                                     "J",
                                     "I",
                                     "9E",
                                     "9J"};
  
  std::vector<std::string> colUnits {"s",
                                     "",
                                     "ADU",
                                     "ADU",
                                     "keV",
                                     "pixel",
                                     "pixel",
                                     "deg",
                                     "deg",
                                     "",
                                     "",
                                     "",
                                     "",
                                     "",
                                     "keV",
                                     "ADU"};
  
  auto table = inFile->addTable("EVENTS", 0, colNames, colForms, colUnits, CCfits::BinaryTbl, 1);
  
  table->addKey("ORIGIN", "ECAP", "Origin of FITS File");
  table->addKey("CREATOR", "SIXTE", "Program that created this FITS file");
  table->addKey("OBS_MODE", "String", "Observation mode");
  table->addKey("DATAMODE", "String", "Instrument data mode");
  table->addKey("MJDREF", 55000, "Modified Julian Date of time origin");
  table->addKey("TIMEZERO", 0.0, "Time correction");
  table->addKey("TIMEUNIT", "s", "Time unit");
  table->addKey("TIMESYS", "TT", "Time system (Terrestial Time)");
  table->addKey("RADECSYS", "FK5", "Stellar reference frame");
  table->addKey("EQUINOX", 2000.0, "Coordinate system equinox");
  table->addKey("LONGSTRN", "OGIP 1.0", "support multi-line COMMENTs or HISTORY records");

  auto& table_pHDU = inFile->pHDU();
  table_pHDU.addKey("ORIGIN", "ECAP", "Origin of FITS File");
  table_pHDU.addKey("CREATOR", "SIXTE", "Program that created this FITS file");

  return inFile;
}

std::unique_ptr<CCfits::FITS> ImpactFileTemplate(const std::string& filename) {
  std::unique_ptr<CCfits::FITS> inFile;
  inFile = std::make_unique<CCfits::FITS>(filename, CCfits::Write);
  
  std::vector<std::string> colNames {"TIME",
                                     "ENERGY",
                                     "X",
                                     "Y",
                                     "PH_ID",
                                     "SRC_ID"};
  
  std::vector<std::string> colForms {"D",
                                     "E",
                                     "D",
                                     "D",
                                     "J",
                                     "J"};
  
  std::vector<std::string> colUnits {"s",
                                     "keV",
                                     "m",
                                     "m",
                                     "",
                                     ""};
  
  auto table = inFile->addTable("IMPACTS", 0, colNames, colForms, colUnits, CCfits::BinaryTbl, 1);
  
  table->addKey("TCTYP3", "RAWX", "");
  table->addKey("TCRPX3", 0., "");
  table->addKey("TCRVL3", 0., "");
  table->addKey("TCDLT3", 1., "");
  table->addKey("TCUNI3", "m", "");
  
  table->addKey("TCTYP4", "RAWY", "");
  table->addKey("TCRPX4", 0., "");
  table->addKey("TCRVL4", 0., "");
  table->addKey("TCDLT4", 1., "");
  table->addKey("TCUNI4", "m", "");
  
  table->addKey("ORIGIN", "ECAP", "Origin of FITS File");
  table->addKey("CREATOR", "SIXTE", "Program that created this FITS file");
  
  table->addKey("TTYPE1", "TIME", "Time of photon impact");
  table->addKey("TTYPE2", "ENERGY", "Energy of the incident photon");
  table->addKey("TTYPE3", "X", "X coordinate of photon impact");
  table->addKey("TTYPE4", "Y", "Y coordinate of photon impact");
  table->addKey("TTYPE5", "PH_ID", "Photon ID");
  table->addKey("TTYPE6", "SRC_ID", "Source ID");
  table->addKey("TTYPE7", "TEL_ID", "Telescope ID");

  auto& table_pHDU = inFile->pHDU();
  table_pHDU.addKey("ORIGIN", "ECAP", "Origin of FITS File");
  table_pHDU.addKey("CREATOR", "SIXTE", "Program that created this FITS file");

  return inFile;
  
  //TODO: Missing: HISTORY(pHDU only) in test -> see if this is set with "normal" run
}

std::unique_ptr<CCfits::FITS> PixImpactFileTemplate(const std::string& filename) {
  std::unique_ptr<CCfits::FITS> inFile;
  inFile = std::make_unique<CCfits::FITS>(filename, CCfits::Write);

  std::vector<std::string> colNames {"TIME",
                                     "ENERGY",
                                     "X",
                                     "Y",
                                     "PH_ID",
                                     "SRC_ID",
                                     "PIXID",
                                     "U",
                                     "V",};

  std::vector<std::string> colForms {"D",
                                     "E",
                                     "D",
                                     "D",
                                     "J",
                                     "J",
                                     "J",
                                     "D",
                                     "D"};

  std::vector<std::string> colUnits {"s",
                                     "keV",
                                     "m",
                                     "m",
                                     "",
                                     "",
                                     "",
                                     "m",
                                     "m"};

  auto table = inFile->addTable("PIXELIMPACT", 0, colNames, colForms, colUnits, CCfits::BinaryTbl, 1);


  table->addKey("ORIGIN", "ECAP", "Origin of FITS File");
  table->addKey("CREATOR", "SIXTE", "Program that created this FITS file");

  table->addKey("TTYPE1", "TIME", "Time of photon impact");
  table->addKey("TTYPE2", "ENERGY", "Energy of the incident photon");
  table->addKey("TTYPE3", "X", "X coordinate of photon impact");
  table->addKey("TTYPE4", "Y", "Y coordinate of photon impact");
  table->addKey("TTYPE5", "PH_ID", "Photon ID");
  table->addKey("TTYPE6", "SRC_ID", "Source ID");
  table->addKey("TTYPE7", "PIXID", "ID of pixel for pixelevent");
  table->addKey("TTYPE8", "U", "Pixel u coordinate of impact");
  table->addKey("TTYPE9", "V", "Pixel v coordinate of impact");

  auto& table_pHDU = inFile->pHDU();
  table_pHDU.addKey("ORIGIN", "ECAP", "Origin of FITS File");
  table_pHDU.addKey("CREATOR", "SIXTE", "Program that created this FITS file");

  return inFile;

  //TODO: Missing: HISTORY(pHDU only) in test -> see if this is set with "normal" run
}

std::unique_ptr<CCfits::FITS> PhotonFileTemplate(const std::string& filename) {
  std::unique_ptr<CCfits::FITS> inFile;
  inFile = std::make_unique<CCfits::FITS>(filename, CCfits::Write);
  
  std::vector<std::string> colNames {"TIME",
                                     "ENERGY",
                                     "RA",
                                     "DEC",
                                     "PH_ID",
                                     "SRC_ID"};
  
  std::vector<std::string> colForms {"D",
                                     "E",
                                     "E",
                                     "E",
                                     "J",
                                     "J"};
  
  std::vector<std::string> colUnits {"s",
                                     "keV",
                                     "deg",
                                     "deg",
                                     "",
                                     ""};
  
  auto table = inFile->addTable("PHOTONS", 0, colNames, colForms, colUnits, CCfits::BinaryTbl, 1);
  
  table->addKey("TCTYP3", "RA---TAN", "");
  table->addKey("TCRPX3", 0., "");
  table->addKey("TCRVL3", 0., "");
  table->addKey("TCDLT3", 1., "");
  table->addKey("TCUNI3", "deg", "");
  
  table->addKey("TCTYP4", "DEC--TAN", "");
  table->addKey("TCRPX4", 0., "");
  table->addKey("TCRVL4", 0., "");
  table->addKey("TCDLT4", 1., "");
  table->addKey("TCUNI4", "deg", "");
  
  table->addKey("ORIGIN", "ECAP", "Origin of FITS File");
  table->addKey("CREATOR", "SIXTE", "Program that created this FITS file");
  
  table->addKey("TTYPE1", "TIME", "Time of photon impact");
  table->addKey("TTYPE2", "ENERGY", "Energy of the incident photon");
  table->addKey("TTYPE3", "RA", "Right ascension of the source");
  table->addKey("TTYPE4", "DEC", "Declination of the source");
  table->addKey("TTYPE5", "PH_ID", "Photon ID");
  table->addKey("TTYPE6", "SRC_ID", "Source ID");
  table->addKey("TTYPE7", "TEL_ID", "Telescope ID");

  auto& table_pHDU = inFile->pHDU();
  table_pHDU.addKey("ORIGIN", "ECAP", "Origin of FITS File");
  table_pHDU.addKey("CREATOR", "SIXTE", "Program that created this FITS file");


  
  //TODO: Missing: HISTORY(pHDU only) in test -> see if this is set with "normal" run
  
  return inFile;
}

} // sixte
