/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include "Polygon.h"

extern "C" {
#include "vector.h"
}

namespace sixte {

class SixteVector {
 public:
  SixteVector(double x, double y, double z)
  : sixte_vector_(x,y,z) {};
  
  SixteVector() = default;
  
  explicit SixteVector(Vector_3 vec)
  : sixte_vector_(std::move(vec)) {};
  
  Vector_3& sixte_vector() {
    return sixte_vector_;
  }
  
  [[nodiscard]] const Vector_3& sixte_vector() const {
    return sixte_vector_;
  }
  
  [[nodiscard]] double x() const {
    return sixte_vector_.x();
  };
  
  [[nodiscard]] double y() const {
    return sixte_vector_.y();
  };
  
  [[nodiscard]] double z() const {
    return sixte_vector_.z();
  };
  
  Vector* c_vector () {
    if (!c_vector_) {
      c_vector_ = Vector {
        .x = x(),
        .y = y(),
        .z = z()
      };
    }
    return &(*c_vector_);
  }
  
 private:
  Vector_3 sixte_vector_;
  
  std::optional<Vector> c_vector_{std::nullopt};
};

class SixtePoint {
public:
  SixtePoint() = default;

  SixtePoint(double x, double y, double z)
  : sixte_point_(x,y,z) {}

  explicit SixtePoint(Point_3 point)
  : sixte_point_(std::move(point)) {}

  [[nodiscard]] Point_3& sixte_point() {
    return sixte_point_;
  }

  [[nodiscard]] const Point_3& sixte_point() const {
    return sixte_point_;
  }

  [[nodiscard]] double x() const {
    return sixte_point_.x();
  };

  [[nodiscard]] double y() const {
    return sixte_point_.y();
  };

  [[nodiscard]] double z() const {
    return sixte_point_.z();
  };

private:
  Point_3 sixte_point_;
};


/// Wrapper class representing a 3D affine transformation using CGAL.
class SixteAffTransformation3 {
public:
  /// Default constructor: creates an identity transformation
  SixteAffTransformation3();

 /// Factory method to create a translation transformation.
 ///
 /// \param vec A vector representing the translation.
 /// \return A SixteAffTransformation3 object representing the translation.
  static SixteAffTransformation3 translation(const SixteVector& vec);

  /// Factory method to create a rotation transformation around the Z-axis.
  ///
  /// \param angle The angle of rotation [rad].
  /// \return A SixteAffTransformation3 object representing the rotation.
  static SixteAffTransformation3 rotationZ(double angle);

  /// Combines this transformation with another transformation.
  ///
  /// \param other Another SixteAffTransformation3 transformation.
  /// \return The combined transformation (this * other)
  SixteAffTransformation3 operator*(const SixteAffTransformation3& other) const;

  /// Applies the transformation to a point.
  ///
  /// \param point The point to transform.
  /// \return The transformed point
  SixtePoint operator()(const SixtePoint& point) const;

  /// Computes the inverse of the transformation.
  ///
  /// \return A SixteAffTransformation3 object representing the inverse
  /// transformation.
  [[nodiscard]] SixteAffTransformation3 inverse() const;

 private:
  /// \brief Private constructor that initializes the transformation with a given CGAL transformation.
  ///
  /// \param trafo A CGAL::Aff_transformation_3 object.
  explicit SixteAffTransformation3(const CGAL::Aff_transformation_3<cgal_Kernel>& trafo);

  /// The underlying CGAL affine transformation.
  CGAL::Aff_transformation_3<cgal_Kernel> trafo_;
};


SixteVector unitVector(double ra, double dec);

SixteVector normalizeVector(const SixteVector& x);

double scalarProduct(const SixteVector& x, const SixteVector& y);

/** Formally:vector_product */
SixteVector crossProduct(const SixteVector& x, const SixteVector& y);

SixteVector vectorDifference(const SixteVector& x2, const SixteVector& x1);

SixteVector interpolateVec(const SixteVector& v1, double t1, const SixteVector& v2, double t2, double time);

SixteVector interpolateCircleVector(const SixteVector& v1, const SixteVector& v2, double phase);

std::pair<double, double> calculateRaDec(const SixteVector& v);

SixteVector operator*(const SixteVector& vec, double a);

SixteVector operator/(const SixteVector& vec, double a);

SixteVector operator*(double a, const SixteVector& vec);

SixteVector operator/(double a, const SixteVector& vec);

SixteVector operator+(const SixteVector& vec1, const SixteVector& vec2);

SixteVector operator-(const SixteVector& vec1, const SixteVector& vec2);

void operator+=(SixteVector& vec1, const SixteVector& vec2);

void operator-=(SixteVector& vec1, const SixteVector& vec2);

void operator==(SixteVector& vec1, const SixteVector& vec2);
} // sixte
