/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "SixteVector.h"

#include "SixteException.h"

namespace sixte {

SixteAffTransformation3::SixteAffTransformation3() : trafo_(CGAL::IDENTITY) {}

SixteAffTransformation3 SixteAffTransformation3::translation(
    const SixteVector& vec) {
  return SixteAffTransformation3(CGAL::Aff_transformation_3<cgal_Kernel>(
      CGAL::Translation(), vec.sixte_vector()));
}

SixteAffTransformation3 SixteAffTransformation3::rotationZ(double angle) {
  double sinrot = sin(angle);
  double cosrot = cos(angle);
  return SixteAffTransformation3(CGAL::Aff_transformation_3<cgal_Kernel>(
      cosrot,  sinrot, 0.,
      -sinrot, cosrot, 0.,
      0.,      0.,     1.,
      1.));
}

SixteAffTransformation3 SixteAffTransformation3::operator*(
    const SixteAffTransformation3& other) const {
  return SixteAffTransformation3(trafo_ * other.trafo_);
}

SixtePoint SixteAffTransformation3::operator()(const SixtePoint& point) const {
  return SixtePoint(trafo_(point.sixte_point()));
}

SixteAffTransformation3 SixteAffTransformation3::inverse() const {
  return SixteAffTransformation3(trafo_.inverse());
}

SixteAffTransformation3::SixteAffTransformation3(
    const CGAL::Aff_transformation_3<cgal_Kernel>& trafo)
    : trafo_(trafo) {}

SixteVector unitVector(double ra, double dec) {
  double cos_dec=cos(dec);
  return {cos_dec*cos(ra), cos_dec*sin(ra), sin(dec)};
}

SixteVector normalizeVector(const SixteVector& x) {
  auto vec_length = x.sixte_vector().squared_length();
  auto d = CGAL::approximate_sqrt(vec_length);
  return SixteVector(x.sixte_vector() / d);
}

double scalarProduct(const SixteVector& x, const SixteVector& y) {
  return CGAL::scalar_product(x.sixte_vector(), y.sixte_vector());
}

SixteVector crossProduct(const SixteVector& x, const SixteVector& y) {
  return SixteVector(CGAL::cross_product(x.sixte_vector(), y.sixte_vector()));
}

SixteVector vectorDifference(const SixteVector& x2, const SixteVector& x1) {
  return SixteVector(x2.sixte_vector()-x1.sixte_vector());
}

SixteVector interpolateVec(const SixteVector& v1, double t1, const SixteVector& v2, double t2, double time) {
  if (t1 > t2) throw SixteException("time 1 is not supposed to be above time 2");
  
  double scaling_factor = (time-t1)/(t2-t1);
  return SixteVector(v1.sixte_vector() + scaling_factor * (v2.sixte_vector()-v1.sixte_vector()));
}

SixteVector interpolateCircleVector(const SixteVector& v1, const SixteVector& v2, double phase) {
  SixteVector x1 = normalizeVector(v1);
  SixteVector x2 = normalizeVector(v2);
  
  if (phase < 0 || phase > 1) throw SixteException("Phase must be a value between 0 and 1");
  
  double cosine_angle = scalarProduct(x1, x2);
  double arcsec_limit = cos(0.1/3600.*M_PI/180.); // 0.1 arcsec.
  
  if (fabs(cosine_angle) < arcsec_limit) {
    // TODO: put in docu:
    // The misalignment between the 2 vectors is more than 0.1 arcsec.
    // This is important to check for the subsequent algorithm,
    // because the vectors should not be aligned parallel or
    // anti-parallel.
    
    double phi = acos(cosine_angle); // Angle between x1 and x2:
    
    // Calculate the second base vector spanning the plane of the circle.
    SixteVector d(x2.sixte_vector()-cosine_angle*x1.sixte_vector());
    x2 = normalizeVector(d);
    
    // Determine the angle corresponding to the phase.
    double sin_phase_phi = sin(phase*phi);
    double cos_phase_phi = cos(phase*phi);
    
    SixteVector r((cos_phase_phi*x1.sixte_vector()) + (sin_phase_phi*x2.sixte_vector()));
    return normalizeVector(r);
  } else {
    // There is quasi no motion at all, so perform a linear interpolation.
    SixteVector r = interpolateVec(x1, 0, x2, 1, phase);
    return normalizeVector(r);
  }
}

std::pair<double, double> calculateRaDec(const SixteVector& v) {
  double dec = asin(v.z()/sqrt(scalarProduct(v, v)));
  double ra = atan2(v.y(), v.x());
  if (ra<0.0) ra += 2.0 * M_PI;
  
  return std::make_pair(ra, dec);
}

SixteVector operator*(const SixteVector& vec, double a) {
  return SixteVector(vec.sixte_vector() * a);
}

SixteVector operator/(const SixteVector& vec, double a) {
  return SixteVector(vec.sixte_vector() / a);
}

SixteVector operator*(double a, const SixteVector& vec) {
  return vec * a;
}

SixteVector operator/(double a, const SixteVector& vec) {
  return vec / a;
}

SixteVector operator+(const SixteVector& vec1, const SixteVector& vec2) {
  return SixteVector(vec1.sixte_vector() + vec2.sixte_vector());
}

SixteVector operator-(const SixteVector& vec1, const SixteVector& vec2) {
  return SixteVector(vec1.sixte_vector() - vec2.sixte_vector());
}

void operator+=(SixteVector& vec1, const SixteVector& vec2) {
  vec1.sixte_vector() += vec2.sixte_vector();
}

void operator-=(SixteVector& vec1, const SixteVector& vec2) {
  vec1.sixte_vector() -= vec2.sixte_vector();
}

} // sixte