/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include "PhotonMetainfo.h"
#include "Polygon.h"
#include "SixteException.h"
#include "SixteVector.h"

#include <memory>
#include <utility>
#include <queue>

extern "C" {
#include "photon.h"
#include "impact.h"
}

namespace sixte {

class SixtePhoton {
 public:
  /**
   * Creates a new photon.
   *
   * @param time               The arrival time of the photon on the absorber [s].
   * @param energy             The photon energy [keV].
   * @param sky_position       Right ascension and declination of photon position [rad]
   * @param detector_position  The position of the photon in the detector coordinate system [m].
   * @param photon_metainfo    The photon metainfo (ID of the photon and ID of the
   *                           originating X-ray source).
   */
  SixtePhoton(double time,
              double energy,
              std::pair<double, double> sky_position,
              const SixtePoint& detector_position,
              const PhotonMetainfo& photon_metainfo)
      : SixtePhoton(time, energy, std::make_optional(sky_position),
                    std::make_optional(detector_position), std::nullopt, photon_metainfo) {}

  SixtePhoton(double time,
              double energy,
              std::pair<double, double> sky_position,
              const PhotonMetainfo& photon_metainfo)
      : SixtePhoton(time, energy, sky_position, std::nullopt, std::nullopt, photon_metainfo) {}

  SixtePhoton(double time,
              double energy,
              const SixtePoint& detector_position,
              const PhotonMetainfo& photon_metainfo)
      : SixtePhoton(time, energy, std::nullopt, detector_position, std::nullopt, photon_metainfo) {}

  SixtePhoton(double time,
              double energy,
              const SixtePoint& detector_position,
              const SixteVector& direction,
              const PhotonMetainfo& photon_metainfo)
      : SixtePhoton(time, energy, std::nullopt, detector_position, direction, photon_metainfo) {}

  explicit SixtePhoton(const Photon& photon)
      : SixtePhoton(photon.time, photon.energy, std::make_pair(photon.ra, photon.dec),
                    PhotonMetainfo(photon.ph_id, photon.src_id)) {}

  SixtePhoton(const Photon& photon, size_t tel_id)
          : SixtePhoton(photon.time, photon.energy, std::make_pair(photon.ra, photon.dec),
                        PhotonMetainfo(photon.ph_id, photon.src_id, tel_id)) {}
                    
  // ToDo: add constructor for SixteVector for RayTracing
  /**
   * Gets the arrival time of the photon.
   *
   * @return    The arrival time of the photon on the absorber [s].
   */
  [[nodiscard]] double time() const {
    return time_;
  };

  /**
   * Gets the photon energy.
   *
   * @return    The photon energy [keV].
   */
  [[nodiscard]] double energy() const {
    return energy_;
  };

  /**
   * Gets the position.
   *
   * @return    The position of the photon in the detector coordinate system [m].
   */
  [[nodiscard]] std::optional<SixtePoint> detector_position() const {
    return detector_position_;
  };

  /**
   * Gets the direction.
   *
   * @return    The direction of the photon.
   */
  [[nodiscard]] std::optional<SixteVector> direction() const {
    return direction_;
  };

  /**
   * Gets the metainfo associated with this photon.
   *
   * @return    The photon metainfo.
   */
  [[nodiscard]] PhotonMetainfo photon_metainfo() const {
    return photon_metainfo_;
  }
  [[nodiscard]] PhotonMetainfo& photon_metainfo_ref() {
    return photon_metainfo_;
  }
  [[nodiscard]] const PhotonMetainfo& photon_metainfo_ref() const {
    return photon_metainfo_;
  }

  [[nodiscard]] std::optional<double> ra() const {
    if (sky_position_) {
      return sky_position_->first;
    } else {
      return std::nullopt;
    }
  }

  [[nodiscard]] std::optional<double> dec() const {
    if (sky_position_) {
      return sky_position_->second;
    } else {
      return std::nullopt;
    }
  }

  // TODO: Would be better to return const Photon* const
  Photon* c_photon() {
    if (!c_photon_) {
      c_photon_ = Photon {
          .time = time(),
          .energy = (float)energy(),
          .ra = sky_position_ ? *ra() : 0,
          .dec = sky_position_ ? *dec() : 0,
          .ph_id = photon_metainfo_.ph_id_,
          .src_id = photon_metainfo_.src_id_,
          .c_tel_id = static_cast<int>(photon_metainfo_.tel_id_)
      };
    }

    return &(*c_photon_);
  }
  
  void setTelID(size_t tel_id);
  
  void setChipID(size_t chip_id);
  
 private:
  SixtePhoton(double time,
              double energy,
              std::optional<std::pair<double, double>> sky_position,
              std::optional<SixtePoint> detector_position,
              std::optional<SixteVector> direction,
              const PhotonMetainfo& photon_metainfo)
      : time_{time},
        energy_{energy},
        sky_position_{std::move(sky_position)},
        detector_position_{std::move(detector_position)},
        direction_{std::move(direction)},
        photon_metainfo_{photon_metainfo} {
    if (!sky_position_ && !detector_position_) {
      throw SixteException("One of the two, sky_position or detector_position, must "
                           "be initialized");
    }
  }

  /// Arrival time of the photon on the absorber [s].
  double time_{};

  /// Photon energy [keV].
  double energy_{};

  /// Right ascension and declination of photon position [rad]
  std::optional<std::pair<double, double>> sky_position_{std::nullopt};

  /// Position of the photon in the detector coordinate system [m].
  std::optional<SixtePoint> detector_position_{std::nullopt};

  /// Position of the photon in the detector coordinate system [m].
  std::optional<SixteVector> direction_{std::nullopt};

  /// Meta information about source photon (ph_id, src_id, tel_id)
  PhotonMetainfo photon_metainfo_{};

  std::optional<Photon> c_photon_{std::nullopt};
};

struct SixtePhotonCmp {
  bool operator() (const SixtePhoton& photon_a, const SixtePhoton& photon_b) const {
    return photon_a.time() > photon_b.time();
  }
};

using SixtePhotons = std::priority_queue<SixtePhoton, std::vector<SixtePhoton>, SixtePhotonCmp>;

SixtePhoton pop_next_photon(SixtePhotons& photons);

}
