/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <memory>
#include <optional>
#include <utility>

#include "NewRMF.h"
#include "XMLData.h"

namespace sixte {

class GradeInfo {

  public:
    GradeInfo(XMLData& xml_data, const XMLNode& grading_node,
              const std::shared_ptr<RmfRegistry>& rmf_registry);

    /**
     * Determine whether the grade can be applied to an impact
     * @param n_pre   Number of samples to the previous impact, nullopt if 
     *                no previous 
     * @param n_post  Number of samples to the next impact, nullopt if no next 

     * @return        true if both n_pre and n_post are larger or equal to
     *                the corresponding grade values
     */
    bool containsImp(
        std::optional<unsigned long> n_pre,
        std::optional<unsigned long> n_post);

    /**
     * Apply RMF for this grade to the given energy
     *
     * @param e_imp   impact energy
     * @return        reconstructed energy as determined by the RMF
     */
    double getSignal(double e_imp);

    unsigned int num() {return num_;};

    unsigned n_post() {return n_post_;}
    unsigned n_pre() {return n_pre_;}

  private:
    unsigned int num_;
    std::string name_;
    unsigned long n_pre_;
    unsigned long n_post_;
    std::shared_ptr<const NewRMF> rmf_;
};

class Grader {
  public:
    Grader(XMLData& xml_data, const std::shared_ptr<RmfRegistry>& rmf_registry);

    /**
     * Simulates the grading process for a given impact
     *
     * @param n_pre   Number of samples to the previous impact, nullopt if
     *                no previous
     * @param n_post  Number of samples to the next impact , nullopt if no next
     * @param e_imp   Impact energy
     * @return        A pair of grade index and reconstructed
     *                energy determined by the grade RMF
     */
    [[nodiscard]] std::pair<std::optional<unsigned int>, double> grade(
        std::optional<unsigned long> n_pre,
        std::optional<unsigned long> n_post,
        double e_imp);

    [[nodiscard]] int getGradeLabel(const std::optional<unsigned int> grade_idx);

    [[nodiscard]] unsigned int get_max_n_post();
    [[nodiscard]] unsigned int get_max_n_pre();
    [[nodiscard]] unsigned int get_num_grades();

  private:
    /**
     * Get the grade index for a given impact information
     *
     * @param n_pre   Number of samples to the previous impact, nullopt if
     *                no previous 
     * @param n_post  Number of samples to the next impact, nullopt if no next 
     * @return        std::nullopt if there is no corresponding grade, else the
     *                index to the first matching entry of grades
     */

    std::optional<unsigned int> getGradeIdx(
        std::optional<unsigned long> n_pre,
        std::optional<unsigned long> n_post);

    std::vector<GradeInfo> grades;
};

}
