/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "SixteGrading.h"
#include "SixteException.h"
#include <algorithm>

namespace sixte {
  GradeInfo::GradeInfo(XMLData& xml_data, const XMLNode& grading_node,
                       const std::shared_ptr<RmfRegistry>& rmf_registry)
      : num_(grading_node.attributeAsInt("num")),
        name_(grading_node.attributeAsString("name")),
        n_pre_(grading_node.attributeAsInt("pre")),
        n_post_(grading_node.attributeAsInt("post")) {

    auto rmf_path = xml_data.dirname() + grading_node.attributeAsString("rmf");
    rmf_registry->load(rmf_path);
    rmf_ = rmf_registry->get(rmf_path);
  }

  bool GradeInfo::containsImp(
      std::optional<unsigned long> n_pre,
      std::optional<unsigned long> n_post) {

    bool pre_ok = !n_pre.has_value() || n_pre.value() >= n_pre_;

    bool post_ok = !n_post.has_value() || n_post.value() >= n_post_;

    return pre_ok && post_ok;
  }

  double GradeInfo::getSignal(double e_imp) {
    auto channel = rmf_->sampleChannel(e_imp);
    if (channel < rmf_->firstChannel()) {
      return 0.;
    }
    return rmf_->sixteGetEBOUNDSEnergy(channel);
  }

  Grader::Grader(XMLData& xml_data, const std::shared_ptr<RmfRegistry>& rmf_registry) {
    auto reconstruction = xml_data.child("detector").child("reconstruction");

    for (auto grading_node: reconstruction.children("grading")) {
      grades.emplace_back(xml_data, grading_node, rmf_registry);
    }
  }

    std::pair<std::optional<unsigned int>, double> Grader::grade(
        std::optional<unsigned long> n_pre,
        std::optional<unsigned long> n_post,
        double e_imp) {
      auto idx = getGradeIdx(n_pre, n_post);

      if (!idx.has_value()) {
        return std::make_pair(idx,0);
      } else {
        return std::make_pair(
            idx,
            grades[idx.value()].getSignal(e_imp)
            );
      }
    }

    int Grader::getGradeLabel(std::optional<unsigned int> grade_idx) {
      if (!grade_idx.has_value()) {
        return -1;
      } else {
        return grades[grade_idx.value()].num();
      }
    }

    std::optional<unsigned int> Grader::getGradeIdx(
        std::optional<unsigned long> n_pre,
        std::optional<unsigned long> n_post) {

      auto it = std::find_if(grades.begin(), grades.end(),
          [n_pre,n_post](GradeInfo& g){return g.containsImp(n_pre, n_post);}
          );

      if (it == grades.end()) {
        return std::nullopt;
      } else {
        return std::distance(grades.begin(), it);
      }
    }

    unsigned int Grader::get_max_n_post() {
      unsigned int out = 0;

      for (GradeInfo& g: grades) {
        out = std::max(out, g.n_post());
      }

      return out;
    }

    unsigned int Grader::get_max_n_pre() {
      unsigned int out = 0;

      for (GradeInfo& g: grades) {
        out = std::max(out, g.n_pre());
      }

      return out;
    }

    unsigned int Grader::get_num_grades() {
      return grades.size();
    }

}
