/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "SixteCrosstalk.h"
#include "SixteException.h"
#include "paraminput.h"
#include "SixteCCFits.h"
#include <CGAL/Kernel/global_functions_2.h>
#include <algorithm>
#include <fstream>
#include <iterator>

namespace sixte {

  CrosstalkOptions::CrosstalkOptions(const std::string opt_str) {
  if (opt_str.compare("all") == 0) {
    therm=true;
    tdm_prop1=true;
    tdm_prop2=true;
    tdm_prop3=true;
    tdm_der=true;
  } else if (opt_str.compare("elec") == 0) {
    tdm_prop1=true;
    tdm_prop2=true;
    tdm_prop3=true;
    tdm_der=true;
  } else if (opt_str.compare("therm") == 0) {
    therm=true;
  } else if (opt_str.compare("tdm_prop") == 0) {
    tdm_prop1=true;
    tdm_prop2=true;
    tdm_prop3=true;
  }
  else if (opt_str.compare("tdm_prop1") == 0) {
    tdm_prop1=true;
  }
  else if (opt_str.compare("tdm_prop2") == 0) {
    tdm_prop2=true;
  }
  else if (opt_str.compare("tdm_prop3") == 0) {
    tdm_prop3=true;
  }
  else if (opt_str.compare("tdm_der") == 0) {
    tdm_der=true;
  }
  else if (opt_str.compare("none") == 0) {
    // options default to false
  }
  else {
    throw SixteException("Cannot recognize doCrossalk option " + opt_str);
  }

  tdm_prop_any = tdm_prop1 || tdm_prop2 || tdm_prop3;
}

VictimMap::VictimMap(unsigned int num_pix)
  : data_{num_pix} {}

void VictimMap::addCoupling(
    const T_PixId i_perp, const T_PixId i_vic,
    const double weight) {

    auto& pix_victims = data_.at(i_perp);

    auto pos = pix_victims.find(i_vic);

    double tot_weight = weight;

    if (pos != pix_victims.end()) {
        // a coupling is already present, so make it stronger
        pos->second += weight;
        tot_weight = pos->second;
    } else {
        pix_victims[i_vic] = weight;
    }

    if (abs(tot_weight) > max_coupling_) {
      max_coupling_ = abs(tot_weight);
    }
}

bool VictimMap::isCoupled(T_PixId i_vic, T_PixId i_perp) {

    auto& pix_victims = data_.at(i_perp);

    return pix_victims.find(i_vic) != pix_victims.end();
}

double VictimMap::getCoupling(T_PixId i_vic, T_PixId i_perp) {

    auto& pix_victims = data_.at(i_perp);

    auto pos = pix_victims.find(i_vic);

    if (pos != pix_victims.end()) {
        return pos->second;
    } else {
        return 0.;
    }
}

std::vector<T_PixId> VictimMap::getVictims(T_PixId i_perp) {

    std::vector<T_PixId> vic_ids;

    for (auto &v: data_.at(i_perp)) {
        vic_ids.push_back(v.first);
    }

    return vic_ids;
}

CrossTalkHandler::CrossTalkHandler(XMLData& xml_data, std::string crosstalk_opt, const ArrayGeometry* const geometry) {
  auto xt_node = xml_data.findNodeByName("crosstalk");

  if (!xt_node) {
    return;
  }

  healog(3) << "\nInitializing crosstalk...\n";

  CrosstalkOptions xt_opt(crosstalk_opt);
  double xt_scaling = queryParameterDouble("CrosstalkScaling");

  // read high level data first
  unsigned int num_pix = xml_data.child("detector").child("geometry").attributeAsInt("npix");
  double dt = 1./xml_data.child("detector").child("readout").attributeAsDouble("samplefreq");

  std::vector<unsigned int> grades_npost;
  auto rec_node = xml_data.findNodeByName("reconstruction");

  if (rec_node) {
    for (auto child: rec_node->children("grading")) {
      grades_npost.push_back(child.attributeAsInt("post"));
    }
  }

  healog(3) << "\nCrosstalk modes switched on:\n";

  // iterate over child nodes to get all crosstalk types
  for (auto child: xt_node->allChildren()) {

    if (std::string("dercrosstalk").compare(child.name()) == 0
        && xt_opt.tdm_der) {
      healog(3) << " - derivative TDM crosstalk\n";

      auto crosstalk = loadDerivCrosstalk(
          xml_data, child, num_pix, grades_npost, dt, xt_scaling);

      xt_types_.push_back(
          std::make_unique<TimeEnergyDepCrosstalk>(
            std::move(crosstalk)
            )
          );

    } else if (std::string("propcrosstalk").compare(child.name()) == 0
        && xt_opt.tdm_prop_any) {
      healog(3) << " - proportional TDM crosstalk\n";

      auto crosstalk = loadPropCrosstalk(
          xml_data, child, num_pix, grades_npost, dt, xt_opt, xt_scaling);

      xt_types_.push_back(
          std::make_unique<TimeEnergyDepCrosstalk>(
            std::move(crosstalk)
            )
          );

    } else if (std::string("thermalcrosstalk").compare(child.name()) == 0
        && xt_opt.therm) {
      healog(3) << " - thermal crosstalk\n";

      std::vector<TimeDepCrosstalk> crosstalks = loadThermalCrosstalks(
          xml_data,
          child,
          num_pix,
          grades_npost,
          geometry,
          xt_scaling);

      for (unsigned int ii=0; ii<crosstalks.size(); ii++) {
        xt_types_.push_back(
            std::make_unique<TimeDepCrosstalk>(
              std::move(crosstalks[ii])
              )
            );
      }
    }
  }

  healog(3) << "\n" << std::endl;

  // initialize high level coupling data
  for (T_PixId i_pix=0; i_pix < num_pix; i_pix++) {

    // get all of the victims of this pixel, for every crosstalk type
    // (use a set as intermediate for easy merging)
    std::set<T_PixId> all_victims;

    for (auto &xt: xt_types_) {
      std::vector<T_PixId> type_victims = xt->getVictims(i_pix);
      for (auto &v: type_victims) {
        all_victims.insert(v);
      }
    }

    victim_lists_.emplace_back(all_victims.begin(), all_victims.end());
    proxy_list_.push_back({});
  }

  for (auto &xt: xt_types_) {
    max_xt_delay_ = std::max(max_xt_delay_, xt->max_xt_delay());
  }
}

const std::vector<T_PixId>& CrossTalkHandler::getVictims(T_PixId i_perp) {
  return victim_lists_.at(i_perp);
}

void CrossTalkHandler::createProxies(T_PixId i_perp, const Signal& perpsig) {
  for (auto &vic: getVictims(i_perp)) {
    proxy_list_[vic].emplace_back(i_perp, perpsig);
  }
}

void CrossTalkHandler::pileupProxies(T_PixId i_perp) {
  for (auto &vic: getVictims(i_perp)) {
    // find the last proxy from this pixel, excluding the last proxy
    auto pos = std::find_if(std::rbegin(proxy_list_[vic])+1, std::rend(proxy_list_[vic]),
        [i_perp](auto p){return p.i_perp() == i_perp;}
        );

    if (pos != std::rend(proxy_list_[vic])) {
      Signal perpsig = proxy_list_[vic].back().perpsig();
      // add the new signal to that proxy
      pos->addSignal(perpsig);
      // remove the last proxy
      proxy_list_[vic].pop_back();
    }
  }
}

std::pair<unsigned int, double> CrossTalkHandler::calcTotalCrosstalk(
    T_PixId i_pix,
    const Signal &pix_signal,
    unsigned int grade_id) {

  cleanProxies(i_pix, pix_signal.creation_time());

  double dE = 0.;
  unsigned int n_xt = 0;

  for (auto& proxy: proxy_list_[i_pix]) {
    for (auto& xt: xt_types_) {
      if (xt->isCoupled(i_pix, proxy.i_perp())) {
        double delay = proxy.perpsig().creation_time() - pix_signal.creation_time();

        dE += xt->computeEnergyShift(i_pix, proxy.i_perp(),
            pix_signal.val(), proxy.perpsig().val(),
            delay, grade_id);
      }
    }
    if (dE != 0.) {n_xt++;}
  }

  return std::pair(n_xt, dE);

}

void CrossTalkHandler::cleanProxiesGlobal(double cutoff_time) {
  size_t num_pix = victim_lists_.size();
  for (size_t i_pix=0; i_pix < num_pix; i_pix++) {
    // use a more conservative figure for the cutoff time here
    cleanProxies(i_pix, cutoff_time-10*max_xt_delay_);
  }
}

void CrossTalkHandler::cleanProxies(T_PixId i_vic, double readout_time) {
  auto& proxies = proxy_list_[i_vic];

  double cutoff = readout_time - max_xt_delay_;

  while (!proxies.empty()) {
    if (proxies.front().perpsig().creation_time() < cutoff) {
      proxies.pop_front();
    } else {
      break;
    }
  }
}

std::vector<T_PixId> CrossTalkHandler::checkProxyTriggers(T_PixId i_perp) {
  std::vector<T_PixId> triggered_pixels;

  for (auto &i_vic: getVictims(i_perp)) {
    // find the last proxy from this pixel
    auto pos = std::find_if(std::rbegin(proxy_list_[i_vic]), std::rend(proxy_list_[i_vic]),
        [i_perp](auto p){return p.i_perp() == i_perp;}
        );

    if (pos != std::rend(proxy_list_[i_vic])) {
      double perp_energy = proxy_list_[i_vic].back().perpsig().val();

      for (auto& xt: xt_types_) {
        if (xt->canTrigger(i_vic, i_perp, perp_energy)) {
          triggered_pixels.push_back(i_vic);
          break;
        }
      }
    }
  }


  return triggered_pixels;
}

TimeEnergyDepCrosstalk::TimeEnergyDepCrosstalk(
        unsigned int num_pix,
        std::vector<unsigned int> grades_npost,
        double dt,
        std::string weight_file,
        double trigger_threshold)
  : coupling_info_{num_pix}, trigger_threshold_{trigger_threshold}
{
    // open input file
    CCfits::FITS intable(weight_file, CCfits::Read, false);

    // get interpolation coordinates
    std::vector<double> perp_grid;
    std::vector<double> vic_grid;
    std::vector<int> offset_grid;

    auto rows = intable.extension("OFFSETS").rows();
    intable.extension("OFFSETS").column("OFFSET").read(
            offset_grid, 1, rows);
    rows = intable.extension("PENERGIES").rows();
    intable.extension("PENERGIES").column("PENERGY").read(
            perp_grid, 1, rows);
    rows = intable.extension("VENERGIES").rows();
    intable.extension("VENERGIES").column("VENERGY").read(
            vic_grid, 1, rows);

    max_evic_ = vic_grid.back();
    max_eperp_ = perp_grid.back();

    // loop over grades
    for (auto npost: grades_npost) {
        // build the extension name
        std::stringstream namestream;
        namestream << "CROSSTALK_"
            << std::setw(5) << std::setfill('0')
            << npost;
        std::string extname = namestream.str();

        // load our data
        std::valarray<double> interp_arr;
        intable.extension(extname).read(interp_arr);
        // copy into vector, because we need raw data access
        std::vector<double> interp_vec(interp_arr.size());
        std::copy(std::begin(interp_arr), std::end(interp_arr), interp_vec.begin());

        // determine time grid
        int naxis1;
        intable.extension(extname).readKey("NAXIS1", naxis1);
        if (naxis1 > int(offset_grid.size())) {
          std::stringstream msg;
          msg << "NAXIS1 of extension " << extname << " is greater than the NAXIS1 of the OFFSETS extension!" << std::endl;
          throw(SixteException(msg.str()));
        }

        std::vector<int> offsets(naxis1); // offsets in samples
        std::copy(offset_grid.begin(), offset_grid.begin()+naxis1, offsets.begin());

        int offset_lo = offsets[0];
        int offset_hi = offsets.back();

        std::vector<double> delays(naxis1); // offsets in seconds
        std::transform(offsets.begin(), offsets.end(), delays.begin(),
                [&dt](int o){return o * dt;});

        dt_lo_.push_back(offset_lo*dt);
        dt_hi_.push_back(offset_hi*dt);

        // build the interpolator
        std::vector<std::vector<double>::iterator > grid_iter_list;
        grid_iter_list.push_back(vic_grid.begin());
        grid_iter_list.push_back(perp_grid.begin());
        grid_iter_list.push_back(delays.begin());

        std::array<int, 3> grid_sizes;
        grid_sizes[0] = vic_grid.size();
        grid_sizes[1] = perp_grid.size();
        grid_sizes[2] = delays.size();

        unsigned int num_elements = std::accumulate(
                grid_sizes.begin(), grid_sizes.end(),
                1, std::multiplies<double>());

        interpolators_.emplace_back(
                grid_iter_list.begin(), grid_sizes.begin(),
                interp_vec.data(), interp_vec.data() + num_elements);

    }
}

void TimeEnergyDepCrosstalk::addCoupling(
    const T_PixId i_perp, const T_PixId i_vic,
    const double weight) {
  coupling_info_.addCoupling(i_perp, i_vic, weight);
}

double TimeEnergyDepCrosstalk::computeEnergyShift(
    T_PixId i_vic, T_PixId i_perp,
    double e_vic, double e_perp,
    double perp_delay, unsigned int grade_id) {

    if (perp_delay < dt_lo_[grade_id]
     || perp_delay > dt_hi_[grade_id]) {
        return 0.;
    }

    // check if e_vic or e_perp are too high
    if (e_vic > max_evic_ || e_perp > max_eperp_) {
      healog(5) << " *** warning: For crosstalk from pixel "
                << i_perp << " to " << i_vic
                << ", one of the energies"
                << "(" << e_perp << ", " << e_vic << ")"
                << " is outside the tabulated values.\n ---> Skipping this event!"
                << std::endl;

      return 0.;
    }

    std::vector<double> coords = {e_vic, e_perp, perp_delay};
    auto res = interpolators_[grade_id].interp(coords.begin());

    return res * getCoupling(i_vic, i_perp);
}

bool TimeEnergyDepCrosstalk::canTrigger(
    T_PixId i_vic, T_PixId i_perp,
    double perp_energy) {

    if (coupling_info_.getMaxCoupling() * perp_energy < trigger_threshold_) {
      // no need to look up this specific pair
      return false;
    }

    return getCoupling(i_vic, i_perp) * perp_energy >= trigger_threshold_;
}

std::vector<T_PixId> TimeEnergyDepCrosstalk::getVictims(T_PixId i_perp) {
  return coupling_info_.getVictims(i_perp);
}

bool TimeEnergyDepCrosstalk::isCoupled(T_PixId i_vic, T_PixId i_perp) {
  return coupling_info_.isCoupled(i_vic, i_perp);
}

double TimeEnergyDepCrosstalk::getCoupling(T_PixId i_vic, T_PixId i_perp) {
  return coupling_info_.getCoupling(i_vic, i_perp);
}

TimeDepCrosstalk::TimeDepCrosstalk(
    unsigned int num_pix,
    std::vector<unsigned int> grades_npost,
    std::string weight_file,
    double trigger_threshold)
  : coupling_info_{num_pix}, trigger_threshold_{trigger_threshold}
{

    CCfits::FITS intable(weight_file, CCfits::Read, false);

    // loop over grades
    for (auto npost: grades_npost) {
        // build the extension name
        std::stringstream namestream;
        namestream << "CROSSTALK_"
            << std::setw(5) << std::setfill('0')
            << npost;
        std::string extname = namestream.str();

        // load our data
        auto &ext = intable.extension(extname);
        unsigned int num_rows = ext.rows();

        std::vector<double> delays(num_rows, 0.);
        std::vector<double> weights(num_rows, 0.);
        ext.column("time_delay").read(delays, 1, num_rows);
        ext.column("weight").read(weights, 1, num_rows);

        dt_lo_.push_back(delays.front());
        dt_hi_.push_back(delays.back());

        // build the interpolator
        std::vector<std::vector<double>::iterator > grid_iter_list;
        grid_iter_list.push_back(delays.begin());

        std::array<int, 1> grid_sizes;
        grid_sizes[0] = num_rows;

        interpolators_.emplace_back(
                grid_iter_list.begin(), grid_sizes.begin(),
                weights.data(), weights.data() + num_rows);
    }

}

void TimeDepCrosstalk::addCoupling(
    const T_PixId i_perp, const T_PixId i_vic,
    const double weight) {
  coupling_info_.addCoupling(i_perp, i_vic, weight);
}

double TimeDepCrosstalk::computeEnergyShift(
    T_PixId i_vic, T_PixId i_perp,
    double /*e_vic*/, double e_perp,
    double perp_delay, unsigned int grade_id) {

    if (perp_delay < dt_lo_[grade_id]
     || perp_delay > dt_hi_[grade_id]
     || e_perp == 0) {
        return 0.;
    }

    std::vector<double> coords = {perp_delay};
    auto res = interpolators_[grade_id].interp(coords.begin());

    return e_perp * res * getCoupling(i_vic, i_perp);

}

bool TimeDepCrosstalk::isCoupled(T_PixId i_vic, T_PixId i_perp) {
  return coupling_info_.isCoupled(i_vic, i_perp);
}

bool TimeDepCrosstalk::canTrigger(
    T_PixId i_vic, T_PixId i_perp,
    double perp_energy) {

    if (coupling_info_.getMaxCoupling() * perp_energy < trigger_threshold_) {
      // no need to look up this specific pair
      return false;
    }

    return getCoupling(i_vic, i_perp) * perp_energy >= trigger_threshold_;
}

std::vector<T_PixId> TimeDepCrosstalk::getVictims(T_PixId i_perp) {
  return coupling_info_.getVictims(i_perp);
}

double TimeDepCrosstalk::getCoupling(T_PixId i_vic, T_PixId i_perp) {
  return coupling_info_.getCoupling(i_vic, i_perp);
}

struct TDMInfo {
  unsigned int pix_id;
  unsigned int row_id;
  unsigned int col_id;
};

std::vector<TDMInfo> parseTDMdat(std::string filename) {
  std::ifstream infile;
  std::string line;

  std::vector<TDMInfo> parsed_data;

  infile.open(filename);
  if (infile.is_open()) {
    while(getline(infile, line))
    {
      std::stringstream ss(line);
      TDMInfo lineinfo;
      ss >> lineinfo.pix_id >> lineinfo.col_id >> lineinfo.row_id;
      parsed_data.push_back(lineinfo);
    }
  }
  infile.close();

  return parsed_data;

}

TimeEnergyDepCrosstalk loadDerivCrosstalk(
    XMLData& xml_data,
    XMLNode& xt_node,
    unsigned int num_pix,
    std::vector<unsigned int> grades_npost,
    double dt,
    double ext_scaling
    ) {

  double scaling = xt_node.attributeAsDouble("scaling");
  double trigger_threshold = xt_node.attributeAsDouble("trig_thresh");
  std::string channel_file = xml_data.child("detector").child("reconstruction").child("crosstalk").child("mux").attributeAsString("channel_freq_list");
  std::string weight_file = xt_node.attributeAsString("filename");

  channel_file = xml_data.dirname() + channel_file;
  weight_file = xml_data.dirname() + weight_file;
  scaling *= ext_scaling;

  // build the object, without coupling
  TimeEnergyDepCrosstalk xtlk(num_pix, grades_npost, dt, weight_file, trigger_threshold);

  // create the coupling
  std::vector<TDMInfo> tdm_info = parseTDMdat(channel_file);

  // deriv xtlk coupling is with the pixels with row N+1 and N-1
  // in the same column

  // for convenience, sort the tdm_info by row and column,
  // such that we start with col 1, row 1, and then ascend row first
  std::sort(tdm_info.begin(), tdm_info.end(),
      [](auto &a, auto &b) {
        if (a.col_id == b.col_id) {
          return a.row_id < b.row_id;
        } else {
          return a.col_id < b.col_id;
        }
      });

  unsigned int row_start_idx = 0;
  for (unsigned int ii=0; ii<tdm_info.size(); ii++) {
    // note: Pixel Ids are zero based, but rows are not!
    auto& current = tdm_info[ii];
    if (current.row_id == 1) {
      row_start_idx = ii;
      continue;
    }

    auto prev_idx = ii - 1;
    auto next_idx = ii + 1;
    if (next_idx >= tdm_info.size()
        || tdm_info[next_idx].col_id != tdm_info[ii].col_id) {
      next_idx = row_start_idx;
    }

    auto& next = tdm_info[next_idx];
    auto& prev = tdm_info[prev_idx];
    xtlk.addCoupling(current.pix_id, prev.pix_id, scaling);
    xtlk.addCoupling(current.pix_id, next.pix_id, scaling);

    // special case to handle row 1, which we skip earlier
    if (prev.row_id == 1) {
      xtlk.addCoupling(prev.pix_id, current.pix_id, scaling);
    }
    if (next.row_id == 1) {
      xtlk.addCoupling(next.pix_id, current.pix_id, scaling);
    }
  }

  return xtlk;
}

TimeEnergyDepCrosstalk loadPropCrosstalk(
    XMLData& xml_data,
    XMLNode& xt_node,
    unsigned int num_pix,
    std::vector<unsigned int> grades_npost,
    double dt,
    CrosstalkOptions xt_opt,
    double ext_scaling
    ) {

  double scaling1 = xt_node.attributeAsDouble("scaling1");
  double scaling2 = xt_node.attributeAsDouble("scaling2");
  double scaling3 = xt_node.attributeAsDouble("scaling3");
  double trigger_threshold = xt_node.attributeAsDouble("trig_thresh");

  std::string channel_file = xml_data.child("detector").child("reconstruction").child("crosstalk").child("mux").attributeAsString("channel_freq_list");
  std::string weight_file = xt_node.attributeAsString("filename");

  channel_file = xml_data.dirname() + channel_file;
  weight_file = xml_data.dirname() + weight_file;

  scaling1 *= ext_scaling;
  scaling2 *= ext_scaling;
  scaling3 *= ext_scaling;

  // build the object, without coupling
  TimeEnergyDepCrosstalk xtlk(
      num_pix, grades_npost, dt, weight_file, trigger_threshold);

  // create the coupling
  std::vector<TDMInfo> tdm_info = parseTDMdat(channel_file);

  // prop xtlk coupling is with:
  // scaling 1: pixel with row N+1
  // scaling 2: all pixels in the column/channel
  // scaling 3: pixel with row N+2

  // for convenience, sort the tdm_info by row and column,
  // such that we start with col 1, row 1, and then ascend row first
  std::sort(tdm_info.begin(), tdm_info.end(),
      [](auto &a, auto &b) {
        if (a.col_id == b.col_id) {
          return a.row_id < b.row_id;
        } else {
          return a.col_id < b.col_id;
        }
      });

  // handle the next and next next pixels, and collect channel info
  unsigned int current_row_start_idx = 0;
  std::vector<unsigned int> row_starts;
  for (unsigned int ii=0; ii<tdm_info.size(); ii++) {
    // note: Pixel Ids are zero based, but rows are not!
    auto& current = tdm_info[ii];
    if (current.row_id == 1) {
      current_row_start_idx = ii;
      row_starts.push_back(ii);
    }

    auto next_idx = ii + 1;
    auto next_next_idx = ii + 2;
    if (next_idx >= tdm_info.size()
        || tdm_info[next_idx].col_id != tdm_info[ii].col_id) {
      next_idx = current_row_start_idx;
      next_next_idx = current_row_start_idx + 1;
    } else if (next_next_idx >= tdm_info.size()
        || tdm_info[next_next_idx].col_id != tdm_info[ii].col_id) {
      next_next_idx = current_row_start_idx;
    }

    auto& next = tdm_info[next_idx];
    auto& next_next = tdm_info[next_next_idx];
    if (xt_opt.tdm_prop1) {
      xtlk.addCoupling(current.pix_id, next.pix_id, scaling1);
    }
    if (xt_opt.tdm_prop3) {
      xtlk.addCoupling(current.pix_id, next_next.pix_id, scaling3);
    }
  }

  // handle the crosstalk with the whole channel
  if (xt_opt.tdm_prop2) {
    for (unsigned int i_chan=0; i_chan<row_starts.size(); i_chan++) {
      unsigned int chan_start = row_starts[i_chan];
      unsigned int chan_stop;
      if (i_chan < row_starts.size()-1) {
        chan_stop = row_starts[i_chan+1]-1;
      } else {
        chan_stop = num_pix - 1;
      }
      for (unsigned int lo_idx=chan_start; lo_idx<chan_stop; lo_idx++) {
        for (unsigned int hi_idx=lo_idx+1; hi_idx<=chan_stop; hi_idx++) {
          xtlk.addCoupling(tdm_info[hi_idx].pix_id, tdm_info[lo_idx].pix_id, scaling2);
          xtlk.addCoupling(tdm_info[lo_idx].pix_id, tdm_info[hi_idx].pix_id, scaling2);
        }
      }
    }
  }

  return xtlk;
}

std::vector<TimeDepCrosstalk> loadThermalCrosstalks(
    XMLData &xml_data,
    XMLNode &xt_node,
    unsigned int num_pix,
    std::vector<unsigned int> grades_npost,
    const ArrayGeometry* const geometry,
    double ext_scaling) {

  std::vector<TimeDepCrosstalk> xtlks;
  std::vector<double> squared_dists;
  std::vector<double> weights;

  for (auto child_node: xt_node.children("pair")) {

    double weight = child_node.attributeAsDouble("weight");
    double distance = child_node.attributeAsDouble("distance");
    double trigger_threshold = child_node.attributeAsDouble("trig_thresh");
    std::string weight_file = child_node.attributeAsString("timedepfile");

    weight *= ext_scaling;
    weight_file = xml_data.dirname() + weight_file;

    squared_dists.push_back(distance*distance);
    weights.push_back(weight);

    // build the crosstalk
    xtlks.emplace_back(num_pix,
        grades_npost,
        weight_file,
        trigger_threshold);
  }

  for (auto pos=squared_dists.begin()+1; pos != squared_dists.end(); pos++) {
    if (*pos < *(pos-1)) {
      throw SixteException("Thermal crosstalk pairings must be given in order of increasing distance!");
    }
  }

  double max_sq_dist = squared_dists.back();

  // get the coupling
  for (T_PixId ii=0; ii<num_pix-1; ii++) {
    Point_2 pos_ii = geometry->getPolygon(ii).center();
    for (T_PixId jj=ii+1; jj<num_pix; jj++) {
      Point_2 pos_jj = geometry->getPolygon(jj).center();
      double pairdist_sq = CGAL::squared_distance(pos_ii, pos_jj);
      if (pairdist_sq > max_sq_dist) {continue;}

      for (unsigned int i_con = 0; i_con < xtlks.size(); i_con++) {
        if (pairdist_sq <= squared_dists[i_con]) {
          xtlks[i_con].addCoupling(ii, jj, weights[i_con]);
          xtlks[i_con].addCoupling(jj, ii, weights[i_con]);
          break;
        }
      }
    }
  }

  return xtlks;
}

}
