/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wshadow"
#pragma GCC diagnostic ignored "-Wunused-parameter"
#pragma GCC diagnostic ignored "-Wunused-but-set-parameter"
#pragma GCC diagnostic ignored "-Wsign-compare"
#include <CCfits>
#pragma GCC diagnostic pop

#include <unordered_map>

#include "NewSIXT.h"
#include "healog.h"
#include "SixteException.h"
#include "TemplateFitsFiles.h"

namespace sixte {

std::unique_ptr<CCfits::FITS> sixteOpenFITSFileRead(const std::string &filename,
                                                    const std::string &file_type);

std::unique_ptr<CCfits::FITS> sixteOpenFITSFileWrite(const std::string &filename,
                                                     bool clobber,
                                                     bool file_template,
                                                     const std::string &file_type);

std::vector<double> get_columns(CCfits::ExtHDU &ext_name, const std::string &col_name, int row);

enum class FileMode {
  read = 0,
  write = 1,
  create_without_clobber = 2,
  create_with_clobber = 3
};

enum class TableType {
  ascii_tbl = ASCII_TBL,
  binary_tbl = BINARY_TBL,
};

enum class ImageType {
  byte_img = BYTE_IMG,
  short_img = SHORT_IMG,
  long_img = LONG_IMG,
  longlong_img = LONGLONG_IMG,
  float_img = FLOAT_IMG,
  double_img = DOUBLE_IMG
};

namespace cfitsio {

void checkCFITSIOStatusThrow(int status, const std::string& error_message);

fitsfile* openFileByMode(const std::string& filename, const FileMode& mode);
fitsfile* createFile(const std::string& filename, bool clobber);
fitsfile* openFile(const std::string& filename, bool open_for_write = false);
void closeFile(fitsfile* fitsfileptr);
void createTable(fitsfile* fitsfile_ptr, TableType table_type, const std::string& extname = "",
                 long naxis2 = 0, int tfields = 0, const std::vector<string>& ttype = {},
                 const std::vector<string>& tfrom = {}, const std::vector<string>& tunit = {});
void createImage(fitsfile* fitsfile_ptr, ImageType image_type, const std::vector<long>& naxes);
void moveToPHDU(fitsfile *fitsfileptr);
void moveToExt(fitsfile* fitsfileptr, const std::string& extname);
void moveToExt(fitsfile *fitsfileptr, int hdu_num);
size_t getColNum(fitsfile* fitsfile_ptr, const std::string& colname);
unsigned long getNumRows(fitsfile* fitsfile_ptr);
size_t getNumCols(fitsfile* fitsfile_ptr);
int getCurrentHduNum(fitsfile* fitsfile_ptr);
void addCol(fitsfile* fitsfile_ptr, size_t colnum, const std::string& ttype, const std::string& tform, const std::string& tunit);

template<typename... T>
constexpr bool always_false = false;

template<typename T>
constexpr int getFitsDataType() {
  if constexpr (std::is_same_v<T, char>) return TSTRING;
  else if constexpr (std::is_same_v<T, unsigned char>) return TBYTE;
  else if constexpr (std::is_same_v<T, short>) return TSHORT;
  else if constexpr (std::is_same_v<T, unsigned short>) return TUSHORT;
  else if constexpr (std::is_same_v<T, int>) return TINT;
  else if constexpr (std::is_same_v<T, unsigned int>) return TUINT;
  else if constexpr (std::is_same_v<T, long>) return TLONG;
  else if constexpr (std::is_same_v<T, long long>) return TLONGLONG;
  else if constexpr (std::is_same_v<T, unsigned long>) return TULONG;
  else if constexpr (std::is_same_v<T, float>) return TFLOAT;
  else if constexpr (std::is_same_v<T, double>) return TDOUBLE;
  else static_assert(always_false<T>, "Unsupported data type");
}

template<typename T>
void fitsUpdateKey(fitsfile* fitsfile_ptr, const std::string& keyname, const T* const value_ptr, const std::string& comment = "") {
  auto data_type = getFitsDataType<T>();
  const char* c_comment = comment.empty() ? nullptr : comment.c_str();
  int status = EXIT_SUCCESS;

  fits_update_key(fitsfile_ptr, data_type, keyname.data(), (void*) value_ptr, c_comment, &status);
  checkStatusThrow(status, "Failed to update keyword " + keyname);
}

void fitsUpdateKeyLongstr(fitsfile* fitsfile_ptr, const std::string& keyname, const std::string& value, const std::string& comment = "");

template<typename T>
void fitsReadKey(fitsfile* fitsfile_ptr, const std::string& keyname, T* const value_ptr) {
  auto data_type = getFitsDataType<T>();
  int status = EXIT_SUCCESS;

  fits_read_key(fitsfile_ptr, data_type, keyname.c_str(), value_ptr, nullptr, &status);
  checkStatusThrow(status, "Failed to read keyword " + keyname);
}

template<typename T>
void fitsWriteCol(fitsfile* fitsfile_ptr, size_t colnum, unsigned long row, unsigned long nelements, const T* const value_ptr) {
  auto data_type = getFitsDataType<T>();
  if (data_type == TSTRING) throw SixteException("Writing string to column not implemented yet");
  int status = EXIT_SUCCESS;

  fits_write_col(fitsfile_ptr, data_type, colnum, row, 1, nelements, (void*) value_ptr, &status);
  checkCFITSIOStatusThrow(status, "Failed to write column data");
}

template<typename T>
void fitsReadCol(fitsfile* fitsfile_ptr, size_t colnum, unsigned long row, unsigned long nelements, T* value_ptr) {
  int data_type = getFitsDataType<T>();
  if (data_type == TSTRING) throw SixteException("Reading string from column not implemented yet");
  int status = EXIT_SUCCESS;
  void* nulval = nullptr;
  int anynul = 0;

  fits_read_col(fitsfile_ptr, data_type, static_cast<int>(colnum), static_cast<long long>(row), 1,
                nelements, nulval, value_ptr, &anynul, &status);
  checkCFITSIOStatusThrow(status, "Failed to read column data");
}

template<typename T>
void fitsReadPix(fitsfile* fitsfile_ptr, const std::vector<long>& fpixel, unsigned long nelements, T* value_ptr) {
  int data_type = getFitsDataType<T>();
  int status = EXIT_SUCCESS;
  void* nulval = nullptr;
  int anynul = 0;

  fits_read_pix(fitsfile_ptr, data_type, const_cast<long*>(fpixel.data()), nelements,
                nulval, value_ptr, &anynul, &status);
  checkCFITSIOStatusThrow(status, "Failed to read pixel data");
}

template<typename T>
void fitsWritePix(fitsfile* fitsfile_ptr, const std::vector<long>& fpixel,
  unsigned long nelements, const T* const value_ptr) {
  int data_type = getFitsDataType<T>();
  int status = EXIT_SUCCESS;

  fits_write_pix(fitsfile_ptr, data_type, const_cast<long*>(fpixel.data()),
                 nelements, (void*) value_ptr, &status);
  checkCFITSIOStatusThrow(status, "Failed to write pixel data");
}

bool checkColExists(fitsfile* fitsfile_ptr, const std::string& colname);

template<typename T>
bool checkKeyExists(fitsfile* fitsfile_ptr, const std::string& key_name, T* const value_ptr) {
  auto data_type = getFitsDataType<T>();
  int status = EXIT_SUCCESS;

  fits_read_key(fitsfile_ptr, data_type, key_name.c_str(), value_ptr, nullptr, &status);

  if (status == EXIT_SUCCESS) return true;
  else return false;
}

} // cfitsio

struct FitsfileDeleter {
  void operator()(fitsfile* fitsfileptr);
};

using FitsfileUniquePtr = std::unique_ptr<fitsfile, FitsfileDeleter>;

class ColNumCache {
public:
  [[nodiscard]] std::optional<size_t> get(int hdu_num, const std::string& colname) const;
  void put(int hdu_num, const std::string& colname, size_t colnum);
  void invalidateHDU(int hdu_num);
  void clear();

private:
  // Cache structure: hdu_num -> (colname -> colnum)
  std::unordered_map<int, std::unordered_map<std::string, size_t>> cache_;
};

class Fitsfile {
 public:
  Fitsfile() = default;
  explicit Fitsfile(const string &filename, const FileMode& mode);
  
  void moveToPHDU();
  void moveToExt(const std::string& extname);
  void moveToExt(int hdu_num);
  size_t getColNum(const std::string& colname);
  unsigned long getNumRows();
  size_t getNumCols();
  void addCol(size_t colnum, const std::string& ttype, const std::string& tform, const std::string& tunit);
  void createTable(TableType table_type, const std::string& extname = "",
                   long naxis2 = 0, int tfields = 0,
                   const std::vector<string>& ttype = {},
                   const std::vector<string>& tfrom = {},
                   const std::vector<string>& tunit = {});
  void createImage(ImageType image_type, const std::vector<long>& naxes, const std::string& extname = "");

  template<typename T>
  void readKey(const string& keyname, T& value);
  template<typename T>
  void updateKey(const string& keyname, const T& value, const string& comment = "");

  template<typename T>
  void readCol(const string& colname, unsigned long row, T& value);

  // Overload for std::vector
  template<typename T>
  void readCol(const string& colname, unsigned long row, unsigned long nelements, std::vector<T>& value);

  template<typename T>
  void writeCol(const string& colname, unsigned long row, const T& value);

  // Overload for std::vector
  template<typename T>
  void writeCol(const string& colname, unsigned long row, unsigned long nelements, const std::vector<T>& value);

  template<typename T>
  void readPix(const std::vector<long>& fpixel, unsigned long nelements, std::vector<T>& value);

  template<typename T>
  void writePix(const std::vector<long>& fpixel, unsigned long nelements, const std::vector<T>& value);

  // TODO: Should be private (currently necessary for old code)
  fitsfile* rawPtr();
  
  bool checkColExists(const std::string& colname);
  template<typename T>
  bool checkKeyExists(const string& keyname, T& value);

 private:
  template<typename T>
  void readKeyImpl(const string& keyname, T& value);
  // Overload for std::string
  void readKeyImpl(const std::string& keyname, std::string& value);

  template<typename T>
  void updateKeyImpl(const string& keyname, const T& value, const string& comment = "");
  // Overload for std::string
  void updateKeyImpl(const string& keyname, const std::string& value, const string& comment = "");
  // Overload for char*
  void updateKeyImpl(const string& keyname, const char* value, const string& comment = "");

  ColNumCache col_num_cache_;
  FitsfileUniquePtr fitsfile_unique_ptr_;
};

template<typename T>
void Fitsfile::readKeyImpl(const string& keyname, T& value) {
  cfitsio::fitsReadKey(rawPtr(), keyname, &value);
}

template<typename T>
void Fitsfile::readKey(const string& keyname, T& value) {
  readKeyImpl(keyname, value);
}

template<typename T>
bool Fitsfile::checkKeyExists(const string& keyname, T& value) {
  return cfitsio::checkKeyExists(rawPtr(), keyname, &value);
}

template<typename T>
void Fitsfile::updateKeyImpl(const string& keyname, const T& value, const string& comment) {
  cfitsio::fitsUpdateKey(rawPtr(), keyname, &value, comment);
}

template<typename T>
void Fitsfile::updateKey(const string& keyname, const T& value, const string& comment) {
  updateKeyImpl(keyname, value, comment);
}

template<typename T>
void Fitsfile::readCol(const string& colname, unsigned long row, T& value) {
  if constexpr (std::is_same_v<T, SixteVector>) {
    std::vector<double> value_vec;
    readCol(colname, row, 3, value_vec);
    value = {value_vec[0], value_vec[1], value_vec[2]};
  } else {
  cfitsio::fitsReadCol(rawPtr(), getColNum(colname), row, 1, &value);
}
}

template<typename T>
void Fitsfile::readCol(const string& colname, unsigned long row, unsigned long nelements, std::vector<T>& value) {
  value.resize(nelements);
  cfitsio::fitsReadCol(rawPtr(), getColNum(colname), row, nelements, value.data());
}

template<typename T>
void Fitsfile::writeCol(const string& colname, unsigned long row, const T& value) {
  if constexpr (std::is_same_v<T, SixteVector>) {
    std::vector<double> value_vec;
    value_vec.push_back(value.x());
    value_vec.push_back(value.y());
    value_vec.push_back(value.z());

    writeCol(colname, row, 3, value_vec);
  } else {
  cfitsio::fitsWriteCol(rawPtr(), getColNum(colname), row, 1, &value);
}
}

template<typename T>
void Fitsfile::writeCol(const string& colname, unsigned long row, unsigned long nelements, const std::vector<T>& value) {
  cfitsio::fitsWriteCol(rawPtr(), getColNum(colname), row, nelements, value.data());
}

template<typename T>
void Fitsfile::readPix(const std::vector<long>& fpixel, unsigned long nelements, std::vector<T>& value) {
  value.resize(nelements);
  cfitsio::fitsReadPix(rawPtr(), fpixel, nelements, value.data());
}

template<typename T>
void Fitsfile::writePix(const std::vector<long>& fpixel, unsigned long nelements, const std::vector<T>& value) {
  cfitsio::fitsWritePix(rawPtr(), fpixel, nelements, value.data());
}

} // sixte