/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2024 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "SixteCCFits.h"

namespace sixte {

std::unique_ptr<CCfits::FITS> sixteOpenFITSFileRead(const std::string &filename, const std::string &file_type) {
  healog(5) << "opening " << file_type << ": " << filename << "..." << std::endl;
  
  std::unique_ptr<CCfits::FITS> inFile;
  inFile = std::make_unique<CCfits::FITS>(filename, CCfits::Read);
  return inFile;
}

std::unique_ptr<CCfits::FITS> sixteOpenFITSFileWrite(const std::string &fits_filename,
                                                     bool clobber,
                                                     bool file_template,
                                                     const std::string &file_type) {
  if (fits_filename.empty()) throw SixteException(file_type + " not specified!");
  healog(5) << "opening " << file_type << ": " << fits_filename << "..." << std::endl;
  
  std::string filename = fits_filename;
  
  if (clobber) filename = "!" + filename;
  
  std::unique_ptr<CCfits::FITS> inFile;
  
  if (file_template) {
    if (file_type == "event file") inFile = EventFileTemplate(filename);
    else if (file_type == "photon file") inFile = PhotonFileTemplate(filename);
    else if (file_type == "impact file") inFile = ImpactFileTemplate(filename);
    else if (file_type == "piximpact file") inFile = PixImpactFileTemplate(filename);
    else throw SixteException("FITS-template not found");
  } else {
    inFile = std::make_unique<CCfits::FITS>(filename, CCfits::Write);
  }
  return inFile;
}

std::vector<double> get_columns(CCfits::ExtHDU &ext_name, const std::string &col_name, int row) {
  std::vector<double> col;
  ext_name.column(col_name).read(col, row);
  return col;
}

void cfitsio::checkCFITSIOStatusThrow(int status, const string &error_message) {
  if (status != EXIT_SUCCESS) {
    char cfitsio_error_message[30]; // max length of cfitsio error msg
    fits_get_errstatus(status, cfitsio_error_message);

    throw SixteException(error_message +
        "\nCFITSIO error: " + std::to_string(status) + " (" + cfitsio_error_message + ")");
  }
}

fitsfile* cfitsio::createFile(const std::string& filename, bool clobber) {
  int status = EXIT_SUCCESS;
  fitsfile* fitsfileptr;

  std::string filename_tmp = filename;
  if (clobber) filename_tmp = "!" + filename_tmp;

  fits_create_file(&fitsfileptr, filename_tmp.c_str(), &status);
  checkCFITSIOStatusThrow(status, "Failed to create file " + filename);

  return fitsfileptr;
}

fitsfile* cfitsio::openFile(const string& filename, bool open_for_write) {
  int status = EXIT_SUCCESS;
  fitsfile* fitsfileptr;
  int iomode = open_for_write ? READWRITE : READONLY;

  fits_open_file(&fitsfileptr, filename.c_str(), iomode, &status);
  checkCFITSIOStatusThrow(status, "Failed to open " + filename + " FITS file");

  return fitsfileptr;
}

fitsfile* cfitsio::openFileByMode(const std::string& filename, const FileMode& mode) {
  switch (mode) {
    case FileMode::read:
      return openFile(filename, false);
    case FileMode::write:
      return openFile(filename, true);
    case FileMode::create_without_clobber:
      return createFile(filename, false);
    case FileMode::create_with_clobber:
      return createFile(filename, true);
    default:
      throw SixteException("Failed to open " + filename);
  }
}

std::vector<char*> convertToCArray(const std::vector<std::string>& vec) {
  std::vector<char*> c_array;
  c_array.reserve(vec.size() + 1);
  for (const auto& str : vec) {
    c_array.push_back(const_cast<char*>(str.c_str()));
  }
  c_array.push_back(nullptr);
  return c_array;
}

void cfitsio::createTable(fitsfile* fitsfile_ptr, TableType table_type, const std::string& extname,
                          long naxis2, int tfields, const std::vector<string>& ttype,
                          const std::vector<string>& tfrom, const std::vector<string>& tunit) {
  int status = EXIT_SUCCESS;

  auto c_ttype = convertToCArray(ttype);
  auto c_tfrom = convertToCArray(tfrom);
  auto c_tunit = convertToCArray(tunit);

  fits_create_tbl(fitsfile_ptr, static_cast<int>(table_type), naxis2, tfields,
                  c_ttype.data(), c_tfrom.data(), c_tunit.data(),
                  extname.c_str(), &status);
  checkCFITSIOStatusThrow(status, "Failed to create table");
}

void cfitsio::createImage(fitsfile* fitsfile_ptr, ImageType image_type, const std::vector<long>& naxes) {
  int status = EXIT_SUCCESS;

  fits_create_img(fitsfile_ptr, static_cast<int>(image_type), static_cast<int>(naxes.size()),
    const_cast<long*>(naxes.data()), &status);
  checkCFITSIOStatusThrow(status, "Failed to create image");
}



void cfitsio::closeFile(fitsfile* fitsfile_ptr) {
  int status = EXIT_SUCCESS;

  fits_close_file(fitsfile_ptr, &status);
  checkCFITSIOStatusThrow(status, "Failed to close file");
}

void cfitsio::moveToPHDU(fitsfile *fitsfileptr) {
  int status = EXIT_SUCCESS;
  
  fits_movabs_hdu(fitsfileptr, 1, nullptr, &status);
  checkCFITSIOStatusThrow(status, "Failed to move to pHDU");
}

void cfitsio::moveToExt(fitsfile* fitsfileptr, const std::string& extname) {
  int status = EXIT_SUCCESS;
  std::string extname_tmp(extname);

  fits_movnam_hdu(fitsfileptr, ANY_HDU, extname_tmp.data(), 1, &status);
  checkCFITSIOStatusThrow(status, "Failed to move to " + extname + " extension");
}

void cfitsio::moveToExt(fitsfile *fitsfileptr, int hdu_num) {
  int status = EXIT_SUCCESS;
  
  // DO NOT REMOVE THE +1 IN HDUNUM! fits_movabs_hdu of 1 would move to the primary hdu!
  fits_movabs_hdu(fitsfileptr, hdu_num+1, nullptr, &status);
  checkCFITSIOStatusThrow(status, "Failed to move to " + std::to_string(hdu_num) + ". extension");
}

size_t cfitsio::getColNum(fitsfile* fitsfile_ptr, const std::string& colname) {
  int status = EXIT_SUCCESS;
  int colnum;
  std::string colname_tmp(colname);
  fits_get_colnum(fitsfile_ptr, CASEINSEN, colname_tmp.data(), &colnum, &status);
  checkCFITSIOStatusThrow(status, "Failed to get column number of " + colname);

  return static_cast<size_t>(colnum);
}

unsigned long cfitsio::getNumRows(fitsfile* fitsfile_ptr) {
  int status = EXIT_SUCCESS;
  long numrows;

  fits_get_num_rows(fitsfile_ptr, &numrows, &status);
  checkCFITSIOStatusThrow(status, "Failed to get number of rows");

  return static_cast<unsigned long>(numrows);
}

size_t cfitsio::getNumCols(fitsfile* fitsfile_ptr) {
  int status = EXIT_SUCCESS;
  int ncols;

  fits_get_num_cols(fitsfile_ptr, &ncols, &status);
  checkCFITSIOStatusThrow(status, "Failed to get number of columns");

  return static_cast<size_t>(ncols);
}

int cfitsio::getCurrentHduNum(fitsfile* fitsfile_ptr) {
  int hdu_num = 0;
  fits_get_hdu_num(fitsfile_ptr, &hdu_num);

  // Convert to 0-based indexing to match SIXTE convention
  // CFITSIO uses 1-based indexing, but SIXTE uses 0-based
  return hdu_num - 1;
}

void cfitsio::addCol(fitsfile* fitsfile_ptr, size_t colnum, const std::string& ttype, const std::string& tform, const std::string& tunit) {
  int status = EXIT_SUCCESS;

  std::string ttype_tmp(ttype);
  std::string tform_tmp(tform);
  std::string tunit_tmp(tunit);

  fits_insert_col(fitsfile_ptr, static_cast<int>(colnum), ttype_tmp.data(), tform_tmp.data(), &status);
  checkCFITSIOStatusThrow(status, "Failed to add column");

  // Set TUNIT of added column
  std::string keyname = "TUNIT" + std::to_string(colnum);
  std::string comment = "Unit of column " + ttype;
  fits_update_key(fitsfile_ptr, TSTRING, keyname.data(), tunit_tmp.data(), comment.data(), &status);
  checkCFITSIOStatusThrow(status, "Failed to update TUNIT of column");
}

void cfitsio::fitsUpdateKeyLongstr(fitsfile* fitsfile_ptr,
                                   const string& keyname,
                                   const string& value,
                                   const string& comment) {
  const char* c_comment = comment.empty() ? nullptr : comment.c_str();
  int status = EXIT_SUCCESS;

  fits_update_key_longstr(fitsfile_ptr, keyname.c_str(), value.c_str(), c_comment, &status);
  checkStatusThrow(status, "Failed to update keyword " + keyname);
}

bool cfitsio::checkColExists(fitsfile* fitsfile_ptr, const std::string& colname) {
  int col_num = 0;
  int status = EXIT_SUCCESS;
  std::string colname_temp = colname;
  
  fits_get_colnum(fitsfile_ptr, CASEINSEN, colname_temp.data(), &col_num, &status);
  if (status == EXIT_SUCCESS) return true;
  else return false;
}

std::optional<size_t> ColNumCache::get(int hdu_num, const std::string& colname) const {
  auto hdu_it = cache_.find(hdu_num);
  if (hdu_it == cache_.end()) {
    return std::nullopt;
  }

  auto col_it = hdu_it->second.find(colname);
  if (col_it == hdu_it->second.end()) {
    return std::nullopt;
  }

  return col_it->second;
}

void ColNumCache::put(int hdu_num, const std::string& colname, size_t colnum) {
  cache_[hdu_num][colname] = colnum;
}

void ColNumCache::invalidateHDU(int hdu_num) {
  cache_.erase(hdu_num);
}

void ColNumCache::clear() {
  cache_.clear();
}

Fitsfile::Fitsfile(const string &filename, const FileMode& mode)
  : fitsfile_unique_ptr_(cfitsio::openFileByMode(filename, mode)) {
  col_num_cache_.clear();
}

void Fitsfile::moveToPHDU() {
  cfitsio::moveToPHDU(rawPtr());
}

void Fitsfile::moveToExt(const string& extname) {
  cfitsio::moveToExt(rawPtr(), extname);
}

void Fitsfile::moveToExt(int hdu_num) {
  cfitsio::moveToExt(rawPtr(), hdu_num);
}

size_t Fitsfile::getColNum(const string& colname) {
  int hdu_num = cfitsio::getCurrentHduNum(rawPtr());

  // Check cache first
  if (auto cached_colnum = col_num_cache_.get(hdu_num, colname)) {
    return *cached_colnum;
  }

  // Cache miss - query and cache result
  size_t colnum = cfitsio::getColNum(rawPtr(), colname);
  col_num_cache_.put(hdu_num, colname, colnum);

  return colnum;
}

unsigned long Fitsfile::getNumRows() {
  return cfitsio::getNumRows(rawPtr());
}

size_t Fitsfile::getNumCols() {
  return cfitsio::getNumCols(rawPtr());
}

void Fitsfile::addCol(size_t colnum, const string& ttype, const string& tform,
                      const string& tunit) {
  cfitsio::addCol(rawPtr(), colnum, ttype, tform, tunit);

  // Invalidate cache for current HDU since structure changed
  col_num_cache_.invalidateHDU(cfitsio::getCurrentHduNum(rawPtr()));
}

void Fitsfile::createTable(TableType table_type, const std::string& extname,
                           long naxis2, int tfields,
                           const std::vector<string>& ttype,
                           const std::vector<string>& tfrom,
                           const std::vector<string>& tunit) {
  cfitsio::createTable(rawPtr(), table_type, extname, naxis2, tfields, ttype, tfrom, tunit);
}

void Fitsfile::createImage(ImageType image_type, const std::vector<long>& naxes, const std::string& extname) {
  cfitsio::createImage(rawPtr(), image_type, naxes);

  if (!extname.empty()) {
    updateKey("EXTNAME", extname, "name of this image extension");
  }
}

void Fitsfile::readKeyImpl(const string& keyname, std::string& value) {
  char value_c_str[MAXMSG];
  cfitsio::fitsReadKey(rawPtr(), keyname, value_c_str);
  value = value_c_str;
}

void Fitsfile::updateKeyImpl(const string& keyname, const string& value, const string& comment) {
  size_t max_std_fitskey_length = 68;
  if (value.size() > max_std_fitskey_length) {
    cfitsio::fitsUpdateKeyLongstr(rawPtr(), keyname, value, comment);
  } else {
    cfitsio::fitsUpdateKey(rawPtr(), keyname, value.c_str(), comment);
  }
}

fitsfile* Fitsfile::rawPtr() {
  return fitsfile_unique_ptr_.get();
}
void Fitsfile::updateKeyImpl(const string& keyname, const char* const value, const string& comment) {
  updateKeyImpl(keyname, std::string(value), comment);
}

void FitsfileDeleter::operator()(fitsfile* fitsfileptr) {
  cfitsio::closeFile(fitsfileptr);
}

bool Fitsfile::checkColExists(const std::string& colname) {
  return cfitsio::checkColExists(rawPtr(), colname);
}


} // sixte