/*
    This file is part of SIXTE.

    SIXTE is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    SIXTE is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    For a copy of the GNU General Public License see
    <http://www.gnu.org/licenses/>.


    Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
    Erlangen-Nuernberg
*/

#include "SixteBackground.h"
#include <cmath>


namespace sixte {

double PHABackground::calcOffaxisAngle(const Point_2& position) const {
  const double r = std::sqrt(position.x()*position.x() + position.y()*position.y());
  // off-axis angle = arctan(detector distance from optical axis / focal length)
  return std::atan2(r, focal_length_);
}
double PHABackground::calcAzimuthAngle(const Point_2& position) {
  double phi = std::atan2(position.y(), position.x());
  if (phi < 0.0) {
    phi += 2.0 * M_PI;
  }
  return phi;
}

PHABackground::PHABackground(const XMLData& xml_data,
                             std::shared_ptr<ArrayGeometry> sensor_geometry,
                             std::shared_ptr<const Ebounds> canonical_ebounds,
                             size_t tel_id,
                             size_t chip_id)
    : sensor_geometry_(std::move(sensor_geometry)),
      tel_id_(tel_id),
      chip_id_(chip_id) {
  // Build an EBOUNDS-only NewRMF
  rmf_ = std::make_shared<NewRMF>(*canonical_ebounds);
  focal_length_ = xml_data.child("telescope").child("focallength").attributeAsDouble("value");

  // check if a phabackground is present in the XML
  if (auto phabkg = xml_data.child("detector").optionalChild("phabackground")) {
    phabkg_.reset(newPHABkgWrapper(
          xml_data.dirname() + phabkg->attributeAsString("filename"),
          *rmf_));
  } else {
    throw SixteException("Background simulation was requested even though no phabackground is given in the XML file!\n       Please either disable the background in your call to SIXTE or add a phabackground to your XML.");
  }

  // Load Vignetting (optional)
  if (auto phabkg = xml_data.child("detector").optionalChild("phabackground")) {
    if (phabkg->hasAttribute("vignetting")) {
      auto vignetting_fullpath = xml_data.dirname() + phabkg->attributeAsString("vignetting");
      vignetting_ = NewVignetting(vignetting_fullpath);
    }
  }

  if (xml_data.root().attributeAsString("telescop") == "THESEUS" &&
    xml_data.root().attributeAsString("instrume") == "SXI") {
    lobster_eye_optic_.emplace(xml_data);
  }
}

CarrierPtr PHABackground::getNextBkgCarrier(const std::pair<double, double>& dt) {
  for(;;) {
    int status = EXIT_SUCCESS;
    double bkg_time;
    long bkg_pha;
    int next_bkg_event = getPHABkgEvent(phabkg_.get(), static_cast<float>(sensor_geometry_->totalSurfaceArea()),
                                        dt.first, dt.second,
                                        &bkg_time, &bkg_pha, &status);
    checkStatusThrow(status, "Failed to generate PHA background event");
    if (!next_bkg_event) {
      return nullptr;
    }

    // Determine the corresponding signal.
    auto energy = rmf_->sixteGetEBOUNDSEnergy(bkg_pha);
    checkStatusThrow(status, "Failed to determine background event energy");

    // Determine position
    auto pix_id = sensor_geometry_->getRandomPixId();
    Point_2 pix_center = sensor_geometry_->getPolygon(pix_id).center();
    SixtePoint position(pix_center.x(), pix_center.y(), 0);

    // If specified, apply vignetting.
    if (vignetting_) {
      // Determine the off-axis angles.
      auto theta = calcOffaxisAngle(pix_center);
      auto phi = calcAzimuthAngle(pix_center);
      // Apply vignetting.
      double p = getUniformRandomNumber();
      if (p > vignetting_->getVignettingFactor(energy, theta, phi)) {
        // The background event is discarded due to vignetting.
        continue;
      }
    }

    PhotonMetainfo photon_metainfo;
    if (lobster_eye_optic_) {
      const Vec3fa origin = lobster_eye_optic_.value().mesh_sensor()[chip_id_].sensorToWorld(static_cast<float>(position.x()) * 1000, static_cast<float>(position.y()) * 1000);
      const SixteVector origin_vec = {origin.x, origin.y, origin.z};
      photon_metainfo = PhotonMetainfo(PhId::pha_bka_id, SrcType::pha_bkg, tel_id_, chip_id_, origin_vec);
    } else {
      photon_metainfo = PhotonMetainfo(PhId::pha_bka_id, SrcType::pha_bkg, tel_id_, chip_id_);
    }

    return std::make_shared<ChargePoint>(EnergyDeposition(energy, position, photon_metainfo, bkg_time));
  }
}

PHABkg* newPHABkgWrapper(const std::string& phabkg_fullpath, const NewRMF& rmf) {
  int status = EXIT_SUCCESS;

  PHABkg* phabkg = newPHABkg(phabkg_fullpath.c_str(), const_cast<struct RMF*>(rmf.rawPtr()), &status);
  checkStatusThrow(status, "Failed to load pha background");

  return phabkg;
}

void PHABkgDeleter::operator()(PHABkg* phabkgptr) const {
  destroyPHABkg(&phabkgptr);
}

}
