/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include "Geometry.h"
#include "SixteCCFits.h"
#include "Carrier.h"
#include "Detector.h"
#include <vector>
#include <gsl/gsl_rng.h>

namespace sixte {

/** Optional: use a Simput light curve to manipulate the background rate over time. */
class BackgroundRateFormLC {
 public:
  /** Use a Simput light curve to manipulate the background rate over time.
   *  The use of this function is optional. If no light curve simput is given
   *  in the xml, this class is not constructed.
   *
   * @param simput_filename Name of simput file containing light curve
   * */
  explicit BackgroundRateFormLC(const std::string& simput_filename);

  /** Determine the background rate over a certain time interval.
   *
   * @param interval Time interval for getting next background rate
   * @return         Vector of rates for the next interval
   * */
  std::vector<double> bkgGetRates(double time_start, double time_end);

 private:
  /** Get the light curve slope for the next time interval.
   *
   * @return Light curve slope
   */
  [[nodiscard]] double getCurrentLCSlope(size_t time, size_t rate) const;

  long num_elements_;

  double start_time_;

  std::vector<double> times_;
  std::vector<double> rates_;
};

class AUXBackground {
 public:
  /** Initialize data structures and read background data table.
   *
   * @param filename            Name of AUX background file
   * @param chip_area           Size of chip area in m2
   * @param aux_rate            (optional) Rate of interacting particles in 1/s
   * @param simput_lc_filename  (optional) Name of simput containing light-curve
   */
  AUXBackground(const std::string& filename,
                double chip_area,
                size_t tel_id,
                size_t chip_id,
                std::optional<double> aux_rate,
                const std::optional<std::string>& simput_lc_filename);

  /** Return a randomly chosen event list for the given interval;
   *  the number of events in the list is determined by poisson statistics
   *  while the choice of the events themselves is based on a flat random
   *  distribution.
   * @param time_start Next attitude time interval start time
   * @param time_end   Next attitude time interval end time
   */
  void bkgGetBackgroundList(double time_start, double time_end);

  /** Return a carrier pointer from buffer or generate a new
   *  random event list for the next attitude dt.
   *
   * @param detector        Detector structure
   * @param photon_metainfo Photon meta-info
   * @param dt              Time interval of current gti bin
   * @param attitude_dt     Attitude time interval
   * @return                Next AUX background carrier
   */
  CarrierPtr getNextAUXBkgCarrier(Detector& detector,
                                  std::pair<double, double> dt,
                                  double attitude_dt);

 private:

  /** @internal
   *  Calculate the number of events for the given interval.
   *
   * @param interval Time interval event rate to be calculated for
   * @return         Event rate for given time interval
   */
  [[nodiscard]] double calcEventRate(double interval) const;

  /** @internal
   *  Write positions of next separate events to an event list.
   *
   *  @return Event list of impacting particles
   */
  std::vector<size_t> fillEventList();

  /** @internal
   *  Checks the unit of the energy column. Currently we only accept 'keV'.
   *  If the unit is 'eV' the module will convert the energy output values
   *  to keV. If no unit is supplied we assume that the unit is keV.
   *  In any other case the module will throw an error and exit.
   *
   *  @param col_num Number of column where energy is specified
   *  @return        Returns true if energy is given in keV and false for eV
   */
  bool energyInputIsKeV(size_t col_num);

  /** Clear output parameters from last interval */
  void clearBackgroundList();

  // The following variables can not be set as a static variable, since the function is initiated more than once
  Carriers aux_photon_carrier_buffer_; /** aux carrier list */
  double time_start_ = 0.; /** start time to generate carriers for next attitude bin */
  double last_time_interval_end_ = 0.; /** end of last time interval carriers created for */

  Fitsfile input_fits_file_; /** FITS file containing aux background data */
  size_t num_rows_input_file_; /** Number of rows in FITS file */
  size_t num_events_{0}; /** Number of total events to be simulated */

  bool is_photon_bkg_{false}; /** Photon or particle background simulated */

  size_t tel_id_; /** telescope id */
  size_t chip_id_; /** detector (chip) id */

  std::vector<double> impact_ids_; /** vector of all impact ids from FITS file */
  std::vector<double> impact_energies_; /** vector of all impact energies from FITS file */
  std::vector<double> impact_x_positions_; /** vector of all impact x positions from FITS file */
  std::vector<double> impact_y_positions_; /** vector of all impact y positions from FITS file */

  std::vector<size_t> event_list_; /** vector containing all indices of events with different times from FITS file */

  double aux_rate_{0.}; /** rate of particles given in xml * chip area */

  bool simput_lc_{false}; /** check if simput containing a light curve is given */

  std::optional<BackgroundRateFormLC> background_rate_form_lc_; /** optional light curve construction */

  // Background output values:
  size_t out_num_events_{0}; /** Number of total events to be simulated + poisson offset */
  size_t out_num_impacts_{0}; /** Number of total sub events to be simulated */

  std::vector<double> out_impact_times_; /** Vector of impact times for given time interval */
  std::vector<double> out_impact_energies_; /** Vector of impact energies for given time interval */
  std::vector<double> out_impact_ids_; /** Vector of impact photon ids for given time interval */

  std::vector<double> out_impact_x_pos_; /** Vector of impact x position for given time interval */
  std::vector<double> out_impact_y_pos_; /** Vector of impact y position for given time interval */
};


} // sixte
