/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <memory>
#include <utility>
#include <vector>
#include "PhotonMetainfo.h"

namespace sixte {

class Signal {
 public:
  Signal() = default;

  Signal(double signal, const PhotonMetainfo& photon_metainfo,
         double creation_time)
      : val_{signal},
        creation_time_{creation_time},
        photon_metainfo_{photon_metainfo} {}

  Signal(double signal, std::vector<PhotonMetainfo>  photon_metainfo,
         double creation_time)
      : val_{signal},
        creation_time_{creation_time},
        photon_metainfo_{std::move(photon_metainfo)} {}

  /**
   * Adds a signal to existing signal. TODO: Update naming.
   *
   * @param signal             The new signal.
   * @param photon_metainfo    The metainfo about source photon.
   */
  void add(const Signal& input_signal) {
    val_ += input_signal.val_;
    photon_metainfo_.insert(std::end(photon_metainfo_),
                            std::begin(input_signal.photon_metainfo_),
                            std::end(input_signal.photon_metainfo_));
  }

  void add(Signal&& input_signal) {
    val_ += input_signal.val_;
    auto& meta = input_signal.photon_metainfo_;
    photon_metainfo_.insert(photon_metainfo_.end(),
                            std::make_move_iterator(meta.begin()),
                            std::make_move_iterator(meta.end()));
    input_signal.reset();
  }

  void scale(double scaling_factor) {
    val_ *= scaling_factor;
  }

  /**
   * Gets the total signal.
   */
  [[nodiscard]] double val() const {
    return val_;
  }

  /**
   * Gets the creation time of the signal.
   */
  [[nodiscard]] double creation_time() const {
    return creation_time_;
  }

  /**
   * Gets the meta information about all source photons.
   */
  [[nodiscard]] const std::vector<PhotonMetainfo>& photon_metainfo() const & {
    return photon_metainfo_;
  }
  [[nodiscard]] std::vector<PhotonMetainfo> photon_metainfo() && {
    return std::move(photon_metainfo_);
  }

  /**
   * Resets the signal.
   */
  void reset() {
    val_ = 0;
    creation_time_ = 0;
    photon_metainfo_.clear();
  }

 private:
  double val_{0};

  double creation_time_{0};

  /// Meta information about source photons (ph_id, src_id)
  std::vector<PhotonMetainfo> photon_metainfo_;
};

}
