/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include "ArrayGeometry.h"
#include "Carrier.h"
#include "Ebounds.h"
#include "GTICollection.h"
#include "LineInfo.h"
#include "ObsInfo.h"
#include "Registry.h"
#include "OperationStrategy.h"
#include "Pixel.h"
#include "Signal.h"
#include "SimulationParameters.h"
#include "XMLData.h"

namespace sixte {

class Absorber;

class Sensor {
 public:
  virtual ~Sensor() = default;

  virtual void addCarrier(const Carrier& carrier) = 0;
  virtual void propagateReadout(double tstop) = 0;
  virtual void finishReadout(double tstop) = 0;
  virtual void clear() = 0;
  virtual void setStartTime(double tstart) = 0;
  virtual void postprocessing(Absorber& absorber,
                              NewAttitude& attitude,
                              const GTICollection& gti) = 0;
  virtual std::shared_ptr<ArrayGeometry> arrayGeometry() = 0;
};

class ShiftArray: public Sensor {
public:
  explicit ShiftArray(XMLData& xml_data, const ObsInfo& obs_info, const OutputFiles& outfiles,
                      bool skip_invalids, size_t xml_id, std::shared_ptr<const Ebounds> canonical_ebounds);

  void addCarrier(const Carrier &carrier) override;
  void propagateReadout(double tstop) override;
  void finishReadout(double tstop) override;
  void clear() override;
  void setStartTime(double tstart) override;
  void postprocessing(Absorber& absorber,
                      NewAttitude& attitude,
                      const GTICollection& gti) override;
  std::shared_ptr<ArrayGeometry> arrayGeometry() override;
  std::shared_ptr<RectangularArray> rectangularArrayGeometry();
  std::shared_ptr<LineInfo> lineInfo();

  void addSignal(const Signal& signal, T_PixId pix_id);
  void addSignal(Signal&& signal, T_PixId pix_id);
  [[nodiscard]] std::optional<Signal> getSignal(T_PixId pix_id) const;
  [[nodiscard]] std::optional<Signal> releaseSignal(T_PixId pix_id);
  void setLastReadoutTime(unsigned int lineindex, double current_clock_time);
  void addLastReadoutTime(unsigned int lineindex, double time);
  void clearPixels(const std::vector<T_PixId>& pix_ids);
  void shiftLines();
  [[nodiscard]] const std::vector<T_PixId>& getPixIdsInLine(unsigned int lineindex) const;
  void clearLine(unsigned int lineindex);
  [[nodiscard]] bool anySignal() const;
  [[nodiscard]] bool anySignalInLine(unsigned int lineindex) const;
  size_t numpix();
  [[nodiscard]] int numLines() const;
  std::pair<unsigned int, unsigned int> PixId2xy(T_PixId pix_id);

private:
  void addActiveID(T_PixId id);
  void removeActiveID(T_PixId id);

  std::shared_ptr<RectangularArray> array_geometry_;
  std::vector<std::unique_ptr<Pixel>> pixels_;
  std::vector<T_PixId> active_pixels_;

  // In a line shift of the pixel array the charges in the shifted pixels are
  // multiplied by this value in order to account for losses due to the shift. TODO
  double charge_transfer_efficiency_{1};

  std::vector<unsigned int> active_count_in_line_;
  int num_lines_;
  int num_pix_in_line_;
  std::vector<std::vector<T_PixId>> pix_ids_in_lines_;
  std::vector<unsigned int> lines_of_pix_ids_;
  std::vector<T_PixId> shift_target_pix_ids_;

  std::shared_ptr<LineInfo> line_info_;
  TimeTriggeredOperation operation_strategy_;
};


class PixArray: public Sensor {
 public:
  PixArray(XMLData& xml_data, const ObsInfo& obs_info, const OutputFiles& outfiles, bool skip_invalids, size_t xml_id, std::shared_ptr<const Ebounds> canonical_ebounds);

  void addCarrier(const Carrier& carrier) override;
  void propagateReadout(double tstop) override;
  void finishReadout(double tstop) override;
  void clear() override;
  void setStartTime(double tstart) override;

  void postprocessing(Absorber& absorber,
                      NewAttitude& attitude,
                      const GTICollection& gti) override;
  std::shared_ptr<ArrayGeometry> arrayGeometry() override; // TODO: Same as in ShiftArray, violates DRY
  std::shared_ptr<RectangularArray> rectangularArrayGeometry();

  [[nodiscard]] std::set<T_PixId> getActiveInArray() const;

  [[nodiscard]] bool anySignal() const;

  [[nodiscard]] const std::optional<Signal>& getSignal(T_PixId pix_id) const;

  [[nodiscard]] std::optional<Signal> releaseSignal(T_PixId pix_id);

  void clearPixels(const std::set<T_PixId>& pix_ids);

  std::pair<unsigned int, unsigned int> PixId2xy(T_PixId pix_id) {
    return array_geometry_->PixId2xy(pix_id);
  }

 private:
  std::shared_ptr<RectangularArray> array_geometry_;
  std::vector<EvtPixel> pixels_;
  std::set<T_PixId> active_pixels_;

  EventTriggeredOperation operation_strategy_;
};

class MicroCal: public Sensor {
  public:
  MicroCal(XMLData& xml_data, const ObsInfo& obs_info, const OutputFiles& outfiles, bool skip_invalids,
           size_t num_xmls, std::shared_ptr<RmfRegistry> rmf_registry);

    void addCarrier(const Carrier& carrier) override;
    void propagateReadout(double tstop) override;
    void finishReadout(double tstop) override;
    void clear() override;
    void setStartTime(double tstart) override;

    void postprocessing(Absorber& absorber,
                        NewAttitude& attitude,
                        const GTICollection& gti) override;

    [[nodiscard]] bool anySignal();
    [[nodiscard]] const std::optional<Signal>& getSignal(T_PixId pix_id);
    void setLastReadoutTime(T_PixId pix_id, double t);
    [[nodiscard]] std::optional<double> getLastReadoutTime(T_PixId pix_id);
    [[nodiscard]] const std::set<T_PixId>& getActivePixels() const;

    void clearSignal(T_PixId pix_id);
    [[nodiscard]] std::pair<unsigned int, double> calcTotalCrosstalk(
        T_PixId i_pix, const Signal& pix_signal, unsigned int grade_id); 

    [[nodiscard]] std::shared_ptr<ArrayGeometry> arrayGeometry() override {return array_geometry_;}

  private:
  template<class Sig>
  void addSignalImpl(const T_PixId& pix_id, Sig&& new_signal) {
    const double new_val  = new_signal.val();
    const double new_time = new_signal.creation_time();

    const auto& pix_sig = getSignal(pix_id);

    if (!pix_sig.has_value()) {
      pixels_[pix_id].setSignal(std::forward<Sig>(new_signal));
    } else {
      // need to check whether we simply add the signal, or read out the pixel
      // and then add the new signal
      const double dt = new_time - pix_sig->creation_time();

      if (dt <= operation_strategy_.t_samp()) {
        // pile up
        pixels_[pix_id].addSignal(std::forward<Sig>(new_signal));
        if (xt_handler_.has_value() && new_val > 0.) {
          xt_handler_->pileupProxies(pix_id);
        }
      } else {
        operation_strategy_.runReadout(*this, pix_id, new_time);
        pixels_[pix_id].setSignal(std::forward<Sig>(new_signal));
      }
    }

    // Remember that this pixel is active.
    active_pixels_.insert(pix_id);
  }

    void addSignal(const T_PixId& pix_id, const Signal& pix_signal);
    void addSignal(const T_PixId& pix_id, Signal&& pix_signal);

    double threshold_event_lo_keV_{0.};

    std::optional<CrossTalkHandler> xt_handler_{std::nullopt};
    size_t proxy_counter{0};
    const size_t PROXY_CLEANUP_PERIOD = 10000; // number of generated proxies after which to perform a cleanup
    std::shared_ptr<ArrayGeometry> array_geometry_;
    std::vector<EvtPixel> pixels_;
    std::set<T_PixId> active_pixels_;
    unsigned int num_events_{0};
    unsigned int num_evt_below_trigger_{0};

    std::vector<std::optional<double>> last_readout_time_; // TODO: Use similar construct as LineInfo in ShiftArray

    MicroCalOperation operation_strategy_;
};

std::unique_ptr<Sensor> createSensor(XMLData& xml_data, const ObsInfo& obs_info,
                                    const OutputFiles& outfiles, bool skip_invalids,
                                    size_t num_xmls, std::shared_ptr<const Ebounds> canonical_ebounds,
                                    std::shared_ptr<RmfRegistry> rmf_registry);

}
