/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2025 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <cstddef>
#include <string>
#include <type_traits>
#include <unordered_map>

#include "SixteException.h"

namespace sixte {

/**
 * @brief Generic lazy-loading keyed registry.
 *
 * Values are created on first load via a Loader functor and then cached.
 *
 * @tparam Key    Key type.
 * @tparam Value  Stored value type.
 * @tparam Loader Functor type with signature Value operator()(const Key&)
 * const.
 */
template <typename Key, typename Value, typename Loader>
class Registry {
 public:
  /// Key type used in the registry.
  using key_type = Key;
  /// Stored value type.
  using mapped_type = Value;
  /// Loader functor type.
  using loader_type = Loader;
  /// Underlying container type.
  using map_type = std::unordered_map<key_type, mapped_type>;

  /**
   * @brief Construct an empty registry.
   */
  Registry() = default;

  /**
   * @brief Construct a registry and reserve space for expected_size entries.
   *
   * @param expected_size Hint for the number of entries to store.
   */
  explicit Registry(std::size_t expected_size) { map_.reserve(expected_size); }

  /**
   * @brief Ensure that the value for key is loaded.
   *
   * If the key already exists, nothing is done. Otherwise the loader is
   * used to create and store a new value.
   *
   * @param key Key whose value should be present in the registry.
   */
  void load(const key_type& key) {
    if (map_.find(key) != map_.end()) {
      return;
    }

    // Construct first, only insert when successful.
    mapped_type value = loader_(key);
    map_.emplace(key, std::move(value));
  }

  /**
   * @brief Get the value for key.
   *
   * @param key Key of the requested value.
   * @return Stored value.
   *
   * @throws SixteException if the key is not present.
   */
  [[nodiscard]] mapped_type get(const key_type& key) const {
    auto it = map_.find(key);
    if (it == map_.end()) {
      throw_not_loaded(key);
    }
    return it->second;
  }

  /**
   * @brief Check if a value for key is stored.
   *
   * @param key Key to look up.
   * @return true if the key is present, false otherwise.
   */
  [[nodiscard]] bool contains(const key_type& key) const {
    return map_.find(key) != map_.end();
  }

  /**
   * @brief Number of stored entries.
   *
   * @return Current size of the registry.
   */
  [[nodiscard]] std::size_t size() const noexcept { return map_.size(); }

  /**
   * @brief Reserve storage for at least n entries.
   *
   * @param min_capacity Minimum number of elements to reserve for.
   */
  void reserve(std::size_t min_capacity) { map_.reserve(min_capacity); }

 private:
  /**
   * @brief Throw a SixteException for a missing key.
   *
   * Uses a more detailed message for std::string keys.
   *
   * @param key Key that was not found.
   *
   * @throws SixteException Always.
   */
  [[noreturn]] void throw_not_loaded(const key_type& key) const {
    if constexpr (std::is_same<key_type, std::string>::value) {
      throw SixteException("Registry::get: value not loaded for key: " + key);
    } else {
      throw SixteException("Registry::get: value not loaded for key");
    }
  }

  /// Loader functor used to create values on demand.
  loader_type loader_{};
  /// Internal map storing key-value pairs.
  map_type map_;
};

}  // namespace sixte
