/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2023 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "RegionFiltering.h"
#include "SixteException.h"

namespace sixte {
SAORegion* getRegion(const std::string &regfile, CCfits::ExtHDU &hdu) {
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wmissing-field-initializers"
  // it seems that wcsdata are supposed to be initialized this way?
  WCSdata wcsdat={.exists=1};
#pragma GCC diagnostic pop
  // check if necessary columns are present
  try {
    hdu.column("X").index();
    hdu.column("Y").index();
  } catch (CCfits::Table::NoSuchColumn &e){
    throw sixte::SixteException("Could not find X or Y columns for region filtering. Have you run 'radec2xy' on your EvtFile?");
  }

  try {
    hdu.readKey("REFXCRPX", wcsdat.xrefpix);
    hdu.readKey("REFYCRPX", wcsdat.yrefpix);
    hdu.readKey("REFXCRVL", wcsdat.xrefval);
    hdu.readKey("REFYCRVL", wcsdat.yrefval);
    hdu.readKey("REFXCDLT", wcsdat.xinc);
    hdu.readKey("REFYCDLT", wcsdat.yinc);
    wcsdat.rot=0.;

    std::string type;
    hdu.readKey("REFXCTYP", type);

    type = type.substr(type.length()-3);
    strcpy(wcsdat.type, ("-" + type).c_str());
  } catch (CCfits::HDU::NoSuchKeyword &e) {
    std::cout << e.message() << std::endl;
    throw sixte::SixteException("Could not read WCS keys for region filtering. Have you run 'radec2xy' on your EvtFile?");
  }

  int status = EXIT_SUCCESS;
  SAORegion *region;
  fits_read_rgnfile(regfile.c_str(), &wcsdat, &region, &status);

  sixte::checkStatusThrow(status, "Failed reading region from regfile");

  return region;
}

bool filterReg(SAORegion *region, double X, double Y) {
  return fits_in_region(X,Y,region);
}
}
