/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <vector>

#include "ArrayGeometry.h"
#include "Ebounds.h"
#include "FrameProcessor.h"
#include "NewEventfile.h"
#include "SixteCrosstalk.h"
#include "SixteGrading.h"
#include "SixteTesEventFile.h"
#include "Registry.h"
#include "sixte_random.h"
extern "C" {
#include "pixeventfile.h"
#include "teseventlist.h"
}

namespace sixte {

class ShiftArray;

class ShiftArrayReadoutStrategy {
 public:
  virtual ~ShiftArrayReadoutStrategy() = default;
  virtual void readoutPixels(ShiftArray& shift_array, unsigned int lineindex,
                             unsigned int readoutindex, double current_clock_time,
                             double readout_time, long frame_number) = 0;

  // TODO: Move outside ReadoutStrategy
  virtual NewEventfile& rawEventFile() = 0;
  virtual const Ebounds& ebounds() const = 0;
};

class SimpleShiftArrayReadout: public ShiftArrayReadoutStrategy {
 public:
  SimpleShiftArrayReadout(XMLData& xml_data, const std::string& rawdata_file, bool clobber,
                          bool delete_rawdata, const ObsInfo& obs_info, std::shared_ptr<const Ebounds> canonical_ebounds);

  ~SimpleShiftArrayReadout() override;

  void readoutPixels(ShiftArray& sensor, unsigned int lineindex,
                     unsigned int readoutindex, double current_clock_time,
                     double readout_time, long frame_number) override;


  const Ebounds& ebounds() const override {
    return *ebounds_;
  }

  NewEventfile& rawEventFile() override {
    return raw_event_file_;
  }

 private:
  std::string rawdata_filename_;
  bool delete_rawdata_;
  NewEventfile raw_event_file_;               // Output event file
  double threshold_readout_lo_keV_{0.}; // Lower readout threshold [keV] (optional).
  std::shared_ptr<const Ebounds> ebounds_;
};


class PixArray;

class PixArrayReadoutStrategy {
 public:
  virtual ~PixArrayReadoutStrategy() = default;
  virtual void readoutPixels(PixArray& pix_array, const std::set<T_PixId>& pix_ids) = 0;

  // TODO: Move outside ReadoutStrategy
  virtual NewEventfile& rawEventFile() = 0;
  [[nodiscard]] virtual const Ebounds& ebounds() const = 0;
};

class SimplePixArrayReadout: public PixArrayReadoutStrategy {
 public:
  SimplePixArrayReadout(XMLData& xml_data, const std::string& rawdata_file, bool clobber,
                        bool delete_rawdata, const ObsInfo& obs_info, std::shared_ptr<const Ebounds> ebounds);

  ~SimplePixArrayReadout() override;

  void readoutPixels(PixArray& pix_array, const std::set<T_PixId>& pix_ids) override;

  NewEventfile& rawEventFile() override {
    return raw_event_file_;
  }

  const Ebounds& ebounds() const override {
    return *ebounds_;
  }

 private:
  std::string rawdata_filename_;
  bool delete_rawdata_;
  NewEventfile raw_event_file_;            // Output event file
  double threshold_readout_lo_keV_{0.}; // Lower readout threshold [keV] (optional).
  std::shared_ptr<const Ebounds> ebounds_;
};



template<typename T_Sensor>
class WFIReadout {
 public:
  using SensorType = T_Sensor;

  WFIReadout(XMLData& xml_data, const std::string& evt_filename, bool clobber, const ObsInfo& obs_info, std::shared_ptr<const Ebounds> ebounds)
      : n_slices_(xml_data.child("detector").child("readoutparams").attributeAsInt("n_slices")),
        pix_per_slice_(xml_data.child("detector").child("readoutparams").attributeAsInt("pix_per_slice")),
        n_lines_(xml_data.child("detector").child("dimensions").attributeAsInt("ywidth")),
        n_pix_(xml_data.child("detector").child("dimensions").attributeAsInt("xwidth")
                   * n_lines_),
        olt_n_frames_(xml_data.child("detector").child("OLT").attributeAsInt("n_frames")),
        nlt_factor_(xml_data.child("detector").child("EFU").attributeAsInt("nlt_factor")),
        elt_factor_(xml_data.child("detector").child("EFU").attributeAsInt("elt_factor")),
        offset_lookup_table_(n_pix_, boost::dynamic_bitset<>(LUT_BANDWIDTH, 0)),
        energy_lookup_table_(n_pix_, boost::dynamic_bitset<>(LUT_BANDWIDTH, 0)),
        neighbor_lookup_table_(n_pix_, boost::dynamic_bitset<>(LUT_BANDWIDTH, 0)),
        asics_(n_slices_, ASIC(xml_data.child("detector").child("ASIC").attributeAsDouble("common_mode_sigma"))),
        adcs_(n_slices_, ADC(xml_data.child("detector").child("ADC").attributeAsDouble("i_min"),
                             xml_data.child("detector").child("ADC").attributeAsDouble("i_max"))),
        frame_processor_(pix_per_slice_, n_slices_, xml_data, evt_filename, clobber, obs_info, std::move(ebounds)) {

    // Init means for offsets
    /*
    auto pixel = xml_data.child("detector").child("pixel");
    double offset_mean = pixel.attributeAsDouble("offset_mean");
    double offset_mean_sigma = pixel.attributeAsDouble("offset_mean_sigma");
    offset_sigma_ = pixel.attributeAsDouble("offset_sigma");

    for (int ii = 0; ii < n_pix_; ii++) {
      double pixel_offset_mean = 0;
      while (pixel_offset_mean <= 0) {
        pixel_offset_mean = offset_mean + getGaussRandomNumber()*offset_mean_sigma;
      }

      pixel_offset_means_.push_back(pixel_offset_mean);
    }
    */
    auto pixel = xml_data.child("detector").child("pixel");
    offsetmap_ = pixel.attributeAsString("offsetmap");
    noisemap_ = pixel.attributeAsString("noisemap");
  }

  void initLookupTables(T_Sensor& sensor) {
    std::cout << "Load offset map" << std::endl;
    pixel_offset_means_ = load_map(sensor, offsetmap_);
    std::cout << "Load noise map" << std::endl;
    pixel_offset_variances_ = load_map(sensor, noisemap_);

    // Adjust pipeline flags
    bool bypass_all_original =
        frame_processor_.commandInterface().pipelineFlag(PipelineFlags::BypassAll);
    bool write_events_original =
        frame_processor_.commandInterface().pipelineFlag(PipelineFlags::WriteEvents);

    frame_processor_.commandInterface().setPipelineFlag(PipelineFlags::BypassAll, true);
    frame_processor_.commandInterface().setPipelineFlag(PipelineFlags::WriteEvents, false);

    // Init LUTs
    std::vector<std::vector<unsigned long>> frame_values(olt_n_frames_,
                                                         std::vector<unsigned long>(n_pix_, 0));

    for (int frame_number = 0; frame_number < olt_n_frames_; frame_number++) {
      // Readout all pixels of this frame
      for (int lineindex = 0; lineindex < n_lines_; lineindex++) {
        readoutPixels(sensor, lineindex, lineindex, 0, frame_number);
      }

      // Store signals in frame_values
      for (int slice_number = 0; slice_number < n_slices_; slice_number++) {
        auto pixel_data_packets = frame_processor_.dataFifo(slice_number);

        for (const auto& packet: pixel_data_packets) {
          //std::cout << "pix id:" << packet.pixel_data.pixID() << ", signal: " << packet.pixel_data.signal().to_ulong() <<std::endl;
          frame_values[frame_number][packet.pixel_data.pixID()] =
              packet.pixel_data.signal().to_ulong();
        }
      }
    }

    //std::cout << "frame_values[1][0] = " << frame_values[1][0] << std::endl;

    // Generate OLT
    // Sum all frames
    std::vector<unsigned long> offset_map(n_pix_, 0);
    for (const auto& frame: frame_values) {
      for (int pix_id = 0; pix_id < n_pix_; pix_id++) {
        offset_map[pix_id] += frame[pix_id];
      }
    }

    // Divide by olt_n_frames
    for (auto& el: offset_map) {
      el /= olt_n_frames_;
    }

    //std::cout << "Offset Map: " << std::endl;
    for (int ii = 0; ii < n_pix_; ii++) {
      //std::cout << "PixID: " << ii << ": " << offset_map[ii] << std::endl;
    }

    // Init OLT
    for (int pix_id = 0; pix_id < n_pix_; pix_id++) {
      offset_lookup_table_[pix_id] = boost::dynamic_bitset<>(LUT_BANDWIDTH, offset_map[pix_id]);
    }
    frame_processor_.commandInterface().setOLT(offset_lookup_table_);

    // Generate noisemap
    std::vector<unsigned long> noise_map(n_pix_, 0);
    for (int pix_id = 0; pix_id < n_pix_; pix_id++) {
      unsigned long squared_sum = 0;
      for (const auto& frame: frame_values) {
        squared_sum += (frame[pix_id] - offset_map[pix_id]) * (frame[pix_id] - offset_map[pix_id]);
      }
      noise_map[pix_id] = sqrt(squared_sum / olt_n_frames_);
    }

    //std::cout << "Noise Map: " << std::endl;
    for (int ii = 0; ii < n_pix_; ii++) {
      //std::cout << "PixID: " << ii << ": " << noise_map[ii] << std::endl;
    }

    // Init ELT
    for (int pix_id = 0; pix_id < n_pix_; pix_id++) {
      energy_lookup_table_[pix_id] = boost::dynamic_bitset<>(LUT_BANDWIDTH, elt_factor_ * noise_map[pix_id]);
      // Add SBs
      energy_lookup_table_[pix_id].push_back(false);
      energy_lookup_table_[pix_id].push_back(true);
    }
    frame_processor_.commandInterface().setELT(energy_lookup_table_);

    //std::cout << "ELT: " << std::endl;
    //for (int ii = 0; ii < n_pix_; ii++) {
    //  std::cout << "PixID: " << ii << ": " << energy_lookup_table_[ii].to_ulong() << std::endl;
    //}

    for (int pix_id = 0; pix_id < n_pix_; pix_id++) {
      neighbor_lookup_table_[pix_id] = boost::dynamic_bitset<>(LUT_BANDWIDTH, nlt_factor_ * noise_map[pix_id]);
      // Add SBs
      neighbor_lookup_table_[pix_id].push_back(false);
      neighbor_lookup_table_[pix_id].push_back(true);
    }
    frame_processor_.commandInterface().setNLT(neighbor_lookup_table_);

    //std::cout << "NLT: " << std::endl;
    //for (int ii = 0; ii < n_pix_; ii++) {
    //  std::cout << "PixID: " << ii << ": " << neighbor_lookup_table_[ii].to_ulong() << std::endl;
    //}

    // Restore pipeline flags
    frame_processor_.commandInterface().setPipelineFlag(PipelineFlags::BypassAll, bypass_all_original);
    frame_processor_.commandInterface().setPipelineFlag(PipelineFlags::WriteEvents, write_events_original);

    frame_processor_.reset();
  }

  LineSignal getLineSignal(const T_Sensor& sensor,
                           unsigned int lineindex) {
    auto pix_ids_in_line = sensor.getPixIdsInLine(lineindex);
    LineSignal line_signal(n_slices_);

    for (int slice_number = 0; slice_number < n_slices_; slice_number++) {
      for (int ii = 0; ii < pix_per_slice_; ii++) {
        T_PixId pix_id = pix_ids_in_line[slice_number * pix_per_slice_ + ii];
        double signal = sensor.getSignal(pix_id).value();
        if (signal > 1e-4) {
          //std::cout << "!!! signal = " << signal << std::endl;
        }
        line_signal[slice_number].emplace_back(pix_id, signal);
        //std::cout << "getLineSignal " << pix_id << ": " << signal << std::endl;
      }
    }

    return line_signal;
  }

  void readoutPixels(T_Sensor& sensor,
                     unsigned int lineindex,
                     unsigned int readoutindex,
                     double readout_time,
                     long frame_number) {
    if (!LUTs_initialized_) {
      std::cout << "\n--- Init Lookup Tables 2 ---" << std::endl;
      LUTs_initialized_ = true;
      initLookupTables(sensor);
      std::cout << "\n--- Lookup Tables Initialized ---" << std::endl;
      //test_run(sensor);
      //throw SixteException("Stop simulation");
    }

    LineSignal line_signal = getLineSignal(sensor, lineindex);
    sensor.clearLine(lineindex);

//    bool broke = false;
//    for (const auto& slice_signal: line_signal) {
//      for (const auto& pixel_signal: slice_signal) {
//        if (pixel_signal.signal > 1e-4) {
//          std::cout << "\n--- Start Signals ---" << std::endl;
//          printLineSignal(line_signal);
//          broke = true;
//          break;
//        }
//      }
//      if (broke)
//        break;
//    }

    addPixelOffsets(line_signal);

    // Run ASIC + ADC
//    LineData adc_out;
//    for (int slice = 0; slice < n_slices_; slice++) {
//      adc_out.push_back(adcs_[slice].processData(
//          asics_[slice].processData(line_signal[slice])));
//    }

    // Run ASIC
    //std::cout << "\n--- Run ASIC ---" << std::endl;
    LineSignal asic_out;
    for (int slice = 0; slice < n_slices_; slice++) {
      asic_out.push_back(asics_[slice].processData(line_signal[slice]));
    }
    //printLineSignal(asic_out);

    // Run ADC
    //std::cout << "\n--- Run ADC ---" << std::endl;
    LineData adc_out;
    for (int slice = 0; slice < n_slices_; slice++) {
      adc_out.push_back(adcs_[slice].processData(asic_out[slice]));
    }
    //printLineData(adc_out);

    // Run frameprocessor
    frame_processor_.processData(adc_out, readoutindex,
                                 readout_time, frame_number);
  }

  void addPixelOffsets(LineSignal& line_signal) {
    for (auto& slice_signal: line_signal) {
      for (auto& pixel_signal: slice_signal) {
        pixel_signal.signal += genPixelOffset(pixel_signal.pix_id);
      }
    }
  }

//  void addPixelOffsets(LineData& line_data) {
//    for (auto& slice_data: line_data) {
//      for (auto& pixel_data: slice_data) {
//        std::cout << "Before: pixel_data " << pixel_data.pixID() << ": " << pixel_data.signal().to_ulong() << std::endl;
//        pixel_data.setSignal( pixel_data.signal().to_ulong() + genPixelOffset(pixel_data.pixID()) );
//        std::cout << "After: pixel_data " << pixel_data.pixID() << ": " << pixel_data.signal().to_ulong() << std::endl;
//      }
//    }
//  }


//  void test_run(T_Sensor& sensor) {
//    std::cout << "\n--- Start Test Run ---" << std::endl;
//
//    // Create a test signal (two slices with five pixels)
//    std::vector<double> testdata_slice1{0, 0, 0, 7, 0};
//    std::vector<double> testdata_slice2{0, 3, 0, 0, 0};
//
//    // Convert to LineSignal
//    LineSignal line_signal;
//    line_signal.push_back(SliceSignal());
//    line_signal.push_back(SliceSignal());
//    for (int id = 0; id < pix_per_slice_; id++) {
//      line_signal[0].emplace_back(id, testdata_slice1[id]);
//      line_signal[1].emplace_back(id + pix_per_slice_, testdata_slice2[id]);
//    }
//
//    std::cout << "\n--- Start Signals ---" << std::endl;
//    printLineSignal(line_signal);
//
//    // Add offset
//    std::cout << "\n--- Add offset ---" << std::endl;
//    addPixelOffsets(line_signal);
//    printLineSignal(line_signal);
//
//    // Run ASIC
//    std::cout << "\n--- Run ASIC ---" << std::endl;
//    LineSignal asic_out;
//    for (int slice = 0; slice < n_slices_; slice++) {
//      asic_out.push_back(asics_[slice].processData(line_signal[slice]));
//    }
//    printLineSignal(asic_out);
//
//    // Run ADC
//    std::cout << "\n--- Run ADC ---" << std::endl;
//    LineData adc_out;
//    for (int slice = 0; slice < n_slices_; slice++) {
//      adc_out.push_back(adcs_[slice].processData(asic_out[slice]));
//    }
//    printLineData(adc_out);
//
//    // Run FrameProcessor
//    std::cout << "\n--- Run FrameProcessor ---" << std::endl;
//    frame_processor_.processData(adc_out, 0, 0, 0);
//
//    std::cout << "\n--- Finish Frame ---" << std::endl;
//    frame_processor_.finishCurrentFrame();
//
//    std::cout << "\n--- Finished Test Run ---" << std::endl;
//  }

  // TODO: Unify interface
  const NewEventfile& eventFile() {
    return frame_processor_.eventFile();
  }

 private:
  std::vector<double> load_map(T_Sensor& sensor, const std::string& filename) {
    int status = EXIT_SUCCESS;
    fitsfile *fptr;
    fits_open_image(&fptr, filename.c_str(), READONLY, &status);

    std::vector<double> map(sensor.numpix());
    long fpixel[2];
    int anynul;
    double val;
    for (int pix_id = 0; pix_id < sensor.numpix(); pix_id++) {
      auto xy_index = sensor.PixId2xy(pix_id);
      fpixel[0] = xy_index.first + 1;
      fpixel[1] = xy_index.second + 1;
      fits_read_pix(fptr, TDOUBLE, fpixel, 1, nullptr, &val, &anynul, &status);
      checkStatusThrow(status, "Failed to load map from " + filename);

      map[pix_id] = val;
    }

    fits_close_file(fptr, &status);
    checkStatusThrow(status, "Failed to close file");

    return map;
  }

  double genPixelOffset(T_PixId pix_id) {
    return pixel_offset_means_[pix_id] + getGaussRandomNumber() * sqrt(pixel_offset_variances_[pix_id]);
  }

  unsigned int n_slices_;
  unsigned int pix_per_slice_;
  unsigned int n_lines_;
  unsigned int n_pix_;

  unsigned int olt_n_frames_;
  unsigned int elt_factor_;
  unsigned int nlt_factor_;

  LookUpTable offset_lookup_table_;
  LookUpTable energy_lookup_table_;
  LookUpTable neighbor_lookup_table_;

  bool LUTs_initialized_{false};

  std::vector<ASIC> asics_;
  std::vector<ADC> adcs_;
  FrameProcessor frame_processor_;

  // Parameters for offset
  std::string offsetmap_;
  std::string noisemap_;
  std::vector<double> pixel_offset_means_;
  std::vector<double> pixel_offset_variances_;
};

class MicroCal;

class MicroCalReadoutStrategy {
 public:
  MicroCalReadoutStrategy(XMLData& xml_data, bool clobber,
      const std::string& evt_filename, const ObsInfo& obs_info,
      std::shared_ptr<RmfRegistry> rmf_registry);

  ~MicroCalReadoutStrategy();

  void readoutPixel(MicroCal& sensor, const T_PixId& pix_id,
      std::optional<unsigned int> n_pre, std::optional<unsigned int> n_post);

  // TODO: Move outside ReadoutStrategy
  TesEventFile *eventFile() {return event_file_;};

  [[nodiscard]] unsigned int max_n_post() const {return max_n_post_;}
  [[nodiscard]] double t_samp() const {return t_samp_;}

 private:
  Grader grader;

  TesEventFile *event_file_;
  double threshold_readout_lo_keV_{0.}; // Lower readout threshold [keV]
  double t_samp_; // sample time
  unsigned int max_n_pre_; // maximum pre record length
  unsigned int max_n_post_; // maximum post record length

  unsigned long num_invalid_evts{0};
  std::vector<unsigned long> num_evts_by_grade;
};

}
