/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#include "ReadoutStrategy.h"

#include "SixteGrading.h"
#include "Sensor.h"
#include "SixteException.h"
#include "SixteTesEventFile.h"
#include "XMLData.h"

namespace sixte {

SimpleShiftArrayReadout::SimpleShiftArrayReadout(XMLData& xml_data, const string& rawdata_file,
                                                 bool clobber, bool delete_rawdata, const ObsInfo& obs_info,
                                                 std::shared_ptr<const Ebounds> ebounds)
    : rawdata_filename_{rawdata_file},
      delete_rawdata_{delete_rawdata},
      raw_event_file_(rawdata_file, clobber, xml_data, obs_info,
                      static_cast<size_t>(ebounds->firstChannel()),
                      ebounds->numberChannels()),
      ebounds_(std::move(ebounds)) {
  if (auto threshold_elem = xml_data.child("detector").optionalChild("threshold_readout_lo_keV")) {
    threshold_readout_lo_keV_ = threshold_elem->attributeAsDouble("value");
  }
  
  // Set event type
  raw_event_file_.updateKeyStringCFITSIO("EVTYPE", "PIXEL", "event type");
}

void SimpleShiftArrayReadout::readoutPixels(ShiftArray& sensor, unsigned int lineindex,
                                            unsigned int readoutindex, double current_clock_time,
                                            double readout_time, long frame_number) {
  sensor.setLastReadoutTime(lineindex, current_clock_time);
  if (!sensor.anySignalInLine(lineindex)) return;

  const auto& pix_ids = sensor.getPixIdsInLine(lineindex);
  const size_t num_pix = pix_ids.size();

  for (size_t ii = 0; ii < num_pix; ++ii) {
    const T_PixId id = pix_ids[ii];
    auto sig = sensor.releaseSignal(id);
    if (!sig) continue;

    if (sig->val() > threshold_readout_lo_keV_) {
      NewEvent evt(*sig, *ebounds_, ii, readoutindex, readout_time, frame_number);
      raw_event_file_.addEvent2CFITSIOFile(evt);
    }
  }
}

SimpleShiftArrayReadout::~SimpleShiftArrayReadout() {
  if (delete_rawdata_) {
    healog(5) << "removing unwanted RawData file: " << rawdata_filename_.c_str() << std::endl;
    int status = remove(rawdata_filename_.c_str());
    if (status != EXIT_SUCCESS) printWarning("removing of RawData file failed");
  }
}

SimplePixArrayReadout::SimplePixArrayReadout(XMLData& xml_data, const string& rawdata_file, bool clobber, bool delete_rawdata, const ObsInfo& obs_info, std::shared_ptr<const Ebounds> ebounds)
    : rawdata_filename_{rawdata_file},
      delete_rawdata_{delete_rawdata},
      raw_event_file_(rawdata_file, clobber, xml_data, obs_info,
                      static_cast<size_t>(ebounds->firstChannel()),
                      ebounds->numberChannels()),
      ebounds_(std::move(ebounds)) {
  if (auto threshold_elem = xml_data.child("detector").optionalChild("threshold_readout_lo_keV")) {
    threshold_readout_lo_keV_ = threshold_elem->attributeAsDouble("value");
  }
}

SimplePixArrayReadout::~SimplePixArrayReadout() {
  if (delete_rawdata_) {
    healog(5) << "removing unwanted RawData file: " << rawdata_filename_.c_str() << std::endl;
    int status = remove(rawdata_filename_.c_str());
    if (status != EXIT_SUCCESS) printWarning("removing of RawData file failed");
  }
}

void SimplePixArrayReadout::readoutPixels(
    PixArray& pix_array,
    const std::set<T_PixId>& pix_ids)
{
  static long frame_number = 0;

  for (auto id: pix_ids) {
    // Get signal and check if it is above the lower readout threshold
    auto signal = pix_array.releaseSignal(id);
    if (!signal) continue;

    if (signal->val() > threshold_readout_lo_keV_) {
      // Add signal to event file.
      auto xyindex = pix_array.PixId2xy(id);
      frame_number++;
      NewEvent event(signal.value(), *ebounds_, xyindex.first, xyindex.second, signal->creation_time(), frame_number);
      raw_event_file_.addEvent2CFITSIOFile(event);
    }
  }
}

MicroCalReadoutStrategy::MicroCalReadoutStrategy(
    XMLData& xml_data, bool clobber,
    const std::string& evt_filename, const ObsInfo& obs_info,
    std::shared_ptr<RmfRegistry> rmf_registry)
  :grader(xml_data, rmf_registry),
   num_evts_by_grade(grader.get_num_grades(),0)
{

  // Get optional readout threshold
  if (auto threshold_elem = xml_data.child("detector").optionalChild("threshold_readout_lo_keV")) {
    threshold_readout_lo_keV_ = threshold_elem->attributeAsDouble("value");
  }

  // Initialize eventfile
  event_file_ = initTesEventFile(evt_filename,clobber,obs_info);

  // timing information
  max_n_pre_ = grader.get_max_n_pre();
  max_n_post_ = grader.get_max_n_post();
  t_samp_ = 1. / xml_data.child("detector").child("readout").attributeAsDouble("samplefreq");
}

MicroCalReadoutStrategy::~MicroCalReadoutStrategy() {
  int status = EXIT_SUCCESS;

  // add event statistics to header
  unsigned long num_valid_evts = std::accumulate(
      num_evts_by_grade.begin(), num_evts_by_grade.end(), 0);
  unsigned long tot_evts = num_invalid_evts + num_valid_evts;

  auto fp = event_file_->fptr;
  char extname[] = "EVENTS";
  fits_movnam_hdu(fp, BINARY_TBL, extname, 1, &status);
  checkStatusThrow(status, "Failed locating EVENTS extension");

  fits_write_key(fp, TULONG, "NEVT", &tot_evts, "number of events simulated", &status);
  fits_write_key(fp, TULONG, "NVALID", &num_valid_evts, "number of events with valid grades", &status);
  fits_write_key(fp, TULONG, "NINVALID", &num_invalid_evts, "number of events with invalid grades", &status);

  for (unsigned int ii=0; ii<num_evts_by_grade.size(); ii++) {
    std::stringstream keyname;
    keyname << "NGRAD" << grader.getGradeLabel(ii);

    std::stringstream comment;
    comment << "number of events with grade " << grader.getGradeLabel(ii);
    fits_write_key(fp, TULONG, keyname.str().c_str(), &num_evts_by_grade[ii], comment.str().c_str(), &status);
  }
  checkStatusThrow(status, "Failed updating TesEventFile header");

  freeTesEventFile(event_file_, &status);
  checkStatusThrow(status, "Failed cleaning up TesEventFile");
}

void MicroCalReadoutStrategy::readoutPixel(
    MicroCal &sensor, const T_PixId &pix_id,
    std::optional<unsigned int> n_pre, std::optional<unsigned int> n_post) {

  const auto& signal = sensor.getSignal(pix_id);

  // apply grading
  auto [grade_idx, graded_energy] = grader.grade(n_pre, n_post, signal->val());

  // get grade1 and grade2
  int grade1, grade2;

  if (n_pre.has_value()) {
      grade1 = (int)n_pre.value();
  } else {
      grade1 = (int)max_n_pre_;
  }

  if (n_post.has_value()) {
      grade2 = (int)n_post.value();
  } else {
      grade2 = (int)max_n_post_;
  }

  double crosstalk_energy = 0.;
  double n_xt = 0;
  if (grade_idx.has_value()) {
      // crosstalk only applies to valid grades
      auto [n, dE] = sensor.calcTotalCrosstalk(
              pix_id, signal.value(), grade_idx.value());

      crosstalk_energy += dE;
      n_xt += n;
  }

  graded_energy += crosstalk_energy;

  // consider events that are either above the readout threshold or invalid
  if ((signal->val() >= threshold_readout_lo_keV_) || !grade_idx.has_value()) {
      // Add signal to event file.
      auto grade_num = grader.getGradeLabel(grade_idx);
      // TODO XRISM CPU limits would go here?

      int status = EXIT_SUCCESS;
      addTesEvent(event_file_, signal->creation_time(), graded_energy,
          pix_id+1, grade1, grade2, grade_num,
          signal->photon_metainfo()[0].ph_id_, signal->photon_metainfo()[0].src_id_,
          n_xt, crosstalk_energy, &status);
      checkStatusThrow(status, "Failed writing TES Event to file");

      // track statistics
      if (grade_idx.has_value()) {
        num_evts_by_grade[grade_idx.value()]++;
      } else {
        num_invalid_evts++;
      }
  }

  sensor.setLastReadoutTime(pix_id, signal->creation_time());

  // clear the pixel signal
  sensor.clearSignal(pix_id);
}

}

