/*
   This file is part of SIXTE.

   SIXTE is free software: you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   any later version.

   SIXTE is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   For a copy of the GNU General Public License see
   <http://www.gnu.org/licenses/>.


   Copyright 2022 Remeis-Sternwarte, Friedrich-Alexander-Universitaet
                  Erlangen-Nuernberg
*/

#pragma once

#include <memory>
#include <string>
#include <vector>
#include "CircularBuffer.h"
#include "ReadoutStrategy.h"
#include "XMLData.h"


namespace sixte {

class ReadoutClock;
class ShiftArray;

class CLElement {
 public:
  virtual ~CLElement() = default;

  virtual bool doOperation(ShiftArray& sensor,
                           ReadoutClock& readout_clock,
                           double tstop) = 0;
};

class CLWait: public CLElement {
 public:
  explicit CLWait(double wait_time)
      : wait_time_{wait_time} {}

  bool doOperation(ShiftArray& sensor,
                   ReadoutClock& readout_clock,
                   double tstop) override final;

 private:
  double wait_time_;
};


class CLLineShift : public CLElement {
 public:
  bool doOperation(ShiftArray& sensor,
                   ReadoutClock& readout_clock,
                   double tstop) override final;
};


class CLNewFrame : public CLElement {
 public:
  bool doOperation(ShiftArray& sensor,
                   ReadoutClock& readout_clock,
                   double tstop) override final;
};


class CLReadoutLine : public CLElement {
 public:
  CLReadoutLine(unsigned int lineindex, unsigned int readoutindex)
      : lineindex_{lineindex}, readoutindex_{readoutindex} {}

  bool doOperation(ShiftArray& sensor,
                   ReadoutClock& readout_clock,
                   double tstop) override final;

 private:
  unsigned int lineindex_;
  unsigned int readoutindex_;
};


class CLClearLine : public CLElement {
 public:
  explicit CLClearLine(unsigned int lineindex)
      : lineindex_{lineindex} {}

  bool doOperation(ShiftArray& sensor,
                   ReadoutClock& readout_clock,
                   double tstop) override final;

 private:
  unsigned int lineindex_;
};


class ReadoutClock {
 public:
  using ClockListElement = CLElement *;

  explicit ReadoutClock(XMLData &xml_data);

  void setReadoutStrategy(std::unique_ptr<ShiftArrayReadoutStrategy> &&strategy);

  [[nodiscard]] double currentClockTime() const;

  void increaseCurrentClockTime(double delta_t);

  void jumpToNextFrame(ShiftArray& sensor, double tstop);

  void readoutLine(ShiftArray &sensor, unsigned int lineindex,
                   unsigned int readoutindex);

  void operate(ShiftArray &sensor, double tstop);
  
  NewEventfile& rawEventFile() {
    return readout_strategy_->rawEventFile();
  }

  const Ebounds& ebounds() const {
    return readout_strategy_->ebounds();
  }

  [[nodiscard]] int rawYMin() const {
    return rawymin_;
  }

  [[nodiscard]] int rawYMax() const {
    return rawymax_;
  }

  [[nodiscard]] double currentFrameStartTime() const {
    return current_frame_start_time_;
  }

  [[nodiscard]] double frameDuration() const {
    return frame_duration_;
  }

  void setStartTime(double tstart);

 private:
  ClockListElement getNewClockListElement(const XMLNode& xml_node);

  CircularBuffer<ClockListElement> clock_list_;
  double current_clock_time_{0.};
  double current_frame_start_time_{0.};
  long current_frame_number_{0};
  double frame_duration_{0.};
  int rawymin_{INT_MAX};
  int rawymax_{0};

  std::unique_ptr<ShiftArrayReadoutStrategy> readout_strategy_;
};

}
